/*
 * Copyright (C) 2005 by egnite Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/*!
 * \file jtagomat.c
 * \brief Main application entry.
 *
 * \verbatim
 *
 * $Log: jtagomat.c,v $
 * Revision 1.4  2006/03/22 19:01:56  haraldkipp
 * Urgent bugfix release.
 *
 * Revision 1.3  2006/03/20 14:14:09  haraldkipp
 * Display version number in verbose mode.
 *
 * Revision 1.2  2005/11/25 13:40:35  haraldkipp
 * Can compile on Linux now.
 *
 * Revision 1.1.1.1  2005/09/14 09:01:09  haraldkipp
 * Initial import.
 *
 *
 * \endverbatim
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef _MSC_VER
#include <windows.h>
#endif

#include <stdlib.h>
#include <string.h>
#include "getopt.h"

#include "jtagomat.h"
#include "tap.h"
#include "ice.h"
#include "parser.h"

#define MAX_DEFINES     32

#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION "1.2.5"
#endif

#include "aztech.h"

int verbose = 0;

/*!
 * \brief Display usage information.
 */
static void usage(void)
{
    fputs("Usage: jtagomat [options] command\n"
          "\nOptions:\n"
          "-a adapter JTAG adapter name\n"
          "-v         Verbose output\n"
          "\nCommands:\n"
          "BATCH file [label]       Run batch file [starting at label]\n"
          "CONTINUE [addr]          Continue target execution [at addr]\n"
          "DEVICE [id]              Read device identifier [and compare]\n"
          "FLASH base [offs [data]] Erase flash sector [and fill with data]\n"
          "HALT                     Stop target execution\n"
          "LOAD addr [num [data]]   Read one [or num] words from addr [and compare to data]\n"
          "MODE [item[=value]]      Show all [or specified] items [or set to value]\n"
          "RESET                    Target hardware reset\n"
          "SAVE addr [data]         Write default [or data] in target addr\n", stderr);
}

/*!
 * \brief JTAG-O-MAT
 *
 * \return 0 on success, otherwise an error code is returned.
 */
int main(int argc, char **argv)
{
    int rc;
    int option;
    char adapter[255];
    int defc = 0;
    char *defv[32];

	LOG_INIT
    adapter[0] = '\0';
    while ((option = getopt(argc, argv, "a:D:v?")) != EOF) {
        switch (option) {
        case 'a':
            strcpy(adapter, optarg);
            break;
        case 'D':
            if (defc >= MAX_DEFINES) {
                fputs("Too many defines\n", stderr);
                return 1;
            }
            defv[defc++] = strdup(optarg);
            break;
        case 'v':
            verbose++;
            break;
        default:
            usage();
            return 1;
        }
    }
    argc -= optind;
    argv += optind;

    if (verbose) {
        fprintf(stderr, "JTAG-O-MAT %s\n", PACKAGE_VERSION);
    }
    if (argc == 0) {
        usage();
        return 1;
    }

    if (TapInit(adapter)) {
        fprintf(stderr, "Failed to initialize %s\n", adapter[0] ? adapter : "standard adapter");
        return 3;
    }
    //IceInit();
    rc = ParseCommand(argc, argv, defc, defv, NULL);
    IceExit();
    TapExit();

	LOG_RELEASE
    return rc;
}
