/* dhcpcCommonLib.c - DHCP client interface shared code library */

/* Copyright 1984 - 1998 Wind River Systems, Inc. */
#include "copyright_wrs.h"

/*
modification history
____________________
01p,06oct98,spm  fixed copying of parameters with IP address pairs (SPR #22416)
01o,04dec97,spm  added code review modifications
01n,06oct97,spm  removed reference to deleted endDriver global; replaced with
                 support for dynamic driver type detection
01m,30sep97,kbw  fixed minor spelling error in library man page
01l,02sep97,spm  modified handling of fatal errors - corrected conditions for
                 disabling network interface and added event hook execution
01k,26aug97,spm  major overhaul: reorganized code and changed user interface
                 to support multiple leases at runtime
01j,06aug97,spm  removed parameters linked list to reduce memory required;
                 renamed class field of dhcp_reqspec structure to prevent C++
                 compilation errors (SPR #9079); corrected minor errors in
                 man pages introduced by 01i revision
01i,30jul97,kbw  fixed man page problems found in beta review
01h,15jul97,spm  cleaned up man pages
01g,10jun97,spm  moved length test to prevent buffer overflow and isolated
                 incoming messages in state machine from input hooks
01f,02jun97,spm  changed DHCP option tags to prevent name conflicts (SPR #8667)
                 and updated man pages
01e,06may97,spm  changed memory access to align IP header on four byte boundary
01d,15apr97,kbw  fixing man page format and wording
01c,07apr97,spm  added code to use Host Requirements defaults and cleanup
                 memory on exit, fixed bugs caused by Ethernet trailers,
                 eliminated potential buffer overruns, rewrote documentation
01b,29jan97,spm  added END driver support and modified to fit coding standards
01a,14nov96,spm  created from shared functions of dhcpcLib.c and dhcpcBootLib.c
*/

/*
DESCRIPTION
This library contains the shared functions used by the run-time version
of the DHCP client both before and after a lease is established.

INCLUDE FILES: dhcpcLib.h

SEE ALSO: dhcpcLib
*/

/* includes */

#include "dhcp/copyright_dhcp.h"
#include "vxWorks.h"
#include "wdLib.h"
#include "semLib.h"
#include "etherLib.h"
#include "rngLib.h"
#include "intLib.h"
#include "vxLib.h" 	/* checksum() declaration */
#include "muxLib.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>


#include "dhcp/dhcpcCommonLib.h"
#include "dhcp/dhcpcStateLib.h"
#include "dhcp/dhcpc.h"
#include "dhcp/dhcpcInternal.h"

#include "app_conf/h/prd_cfg.h"

/* defines */

#define _BYTESPERWORD 	4 	/* Conversion factor for IP header */

     /* Host requirements documents default values. */

#define _HRD_MAX_DGRAM  576     /* Default maximum datagram size. */
#define _HRD_IP_TTL     64      /* Default IP time-to-live (seconds) */
#define _HRD_MTU        576     /* Default interface MTU */
#define _HRD_ROUTER     0xffffffff /* Default router solication */
                                   /* address - 255.255.255.255 */
#define _HRD_ARP_TIME   60      /* Default ARP cache timeout (seconds) */
#define _HRD_TCP_TTL    64      /* Default TCP time-to-live (seconds) */
#define _HRD_TCP_TIME   7200    /* Default TCP keepalive interval (seconds) */

/* externals */

IMPORT SEM_ID  dhcpcMutexSem;    /* Protects the DHCP status indicator */
IMPORT void	   *pDhcpcBootCookie[];

/* globals */

struct interface 	dhcpif;

WDOG_ID		dhcpcEventTimer; 	/* Timeout for DHCP events */
RING_ID		dhcpcEventRing; 	/* Ring buffer of DHCP events */
RING_ID		dhcpcMsgRing; 		/* Ring buffer of DHCP messages */
SEM_ID		dhcpcEventSem; 		/* DHCP event notification */
struct msg 	dhcpcIfMsg; 		/* Incoming DHCP message components */
char * 		pDhcpcRecvBuf; 		/* Message buffer for state machine. */
SEM_ID		dhcpcArpEventSem; 	/* ARP reply notification */
u_char		dhcpArpSpa [4]; /* Sender protocol address for ARP replies */
struct if_info	dhcpcDevice; 		/* Specifies underlying hardware */

/* locals */

LOCAL char inputBuffer [ETHERHL + sizeof (struct ether_arp)]; /* ARP replies */

/* forward declarations */

void dhcpcMon (void);                              /* Monitor lease */
BOOL dhcpcInputHook (struct ifnet *, char *, int);
LOCAL STATUS dhcpcEventGet (EVENT_DATA *);
LOCAL STATUS dhcpcEventHandle (EVENT_DATA *);
char *dhcpcOptionFind (LEASE_DATA *, int, int *); 	/* Locate option */
LOCAL void dhcpcCleanup (void);


extern SEM_ID dhcpc_SemId;

/*******************************************************************************
*
* dhcpcMon - monitor all DHCP client activity
*
* This routine establishes and monitors all DHCP leases. It receives all DHCP
* event notifications (message arrivals, timeouts, and user requests), and
* controls the event processing by invoking the appropriate routines.
* It is the entry point for the monitor task created during the library
* initialization and should only be called internally.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void dhcpcMon (void)
    {
    int offset;
    int level;
    struct ifnet * pIf;
    EVENT_DATA 	newEvent;
    STATUS 	result;
    BOOL 	bsdDrvFlag = TRUE;
	int		i;
	short	num_wan;

    /*
     * Intercept all incoming DHCP messages for any interface. Currently,
     * the same input hook routine must be added once for all BSD drivers,
     * if any, and once for each END-style driver, since the library cannot
     * anticipate which interfaces will be used. Eventually we will be able
     * to register DHCP as a protocol and eliminate the use of input hooks.
     * At that point, we won't need this messy access to the global interface
     * list.
     */

    level = splnet ();
    for (pIf = ifnet; pIf != NULL; pIf = pIf->if_next)
        {
        if (strcmp (pIf->if_name, "lo") == 0)   /* Skip loopback interfaces. */
            continue;


        if ( dhcpc_cfg_IsDhcpClient( pIf->if_name, pIf->if_unit ) == FALSE )
             continue;
#if 0

        if ( Is_WebRedirect_On() == FALSE )
        {
            if ( ( strcmp(pIf->if_name, "mac") == 0 ) &&
                 ( pIf->if_unit == 0 ) )
            {
            
                 continue;                           /* Skip ethernet (LAN) interface */
            }
        }   
          
#endif
        if (muxDevExists (pIf->if_name, pIf->if_unit))
            etherInputHookAdd (dhcpcInputHook, pIf->if_name, pIf->if_unit);
        else if (bsdDrvFlag)
            {
            /* Only set one input hook for all BSD drivers. */
            etherInputHookAdd (dhcpcInputHook, NULL, pIf->if_unit);
            bsdDrvFlag = FALSE;
            }
        }
    splx (level);

    semGive(dhcpc_SemId);
    
    /* Wait for WAN Link is up before starting the FSM. */
    #ifndef OEM_WEBREDIRECT
	while(!WanLinkReady())
		taskDelay(sysClkRateGet() / 10);
    #endif
    
    FOREVER
        {
        /* Wait for an incoming DHCP message, user request, or timeout. */

        result = dhcpcEventGet (&newEvent);
        if (result == ERROR)
            {
            /* Invalid or missing event - ignore. */

            continue;
            }

        /* Handle dhcpcShutdown() requests. */

        if (newEvent.source == DHCP_USER_EVENT &&
                newEvent.type == DHCP_USER_SHUTDOWN)
            {
            /* Release all active leases, free all memory, and exit. */

            newEvent.type = DHCP_USER_RELEASE;

            for (offset = 0; offset < dhcpcMaxLeases; offset++)
                {
                if (dhcpcLeaseList [offset] != NULL)
                    {
                        newEvent.leaseId = dhcpcLeaseList [offset];
                        dhcpcEventHandle (&newEvent);

                        /* clear the Cookie point.*/
						SYSTEM_NumOfWanGet(&num_wan);
                        for (i=0; i < num_wan; i++)
                            if (pDhcpcBootCookie[i] == dhcpcLeaseList [offset])
                            {
                                pDhcpcBootCookie[i] = NULL;
                                break;
                            }

                        dhcpcLeaseList [offset] = NULL;
                    }
                }
            dhcpcCleanup ();
            free(dhcpcLeaseList); 
            dhcpcLeaseList = NULL;
            dhcpcInitialized = FALSE;
            break;
            }

        /* Process all other events in the context of the target lease. */
        result = dhcpcEventHandle (&newEvent);
        if (result == DHCPC_DONE)
            {
            /* Set the list entry to NULL when a lease is removed. */

            for (offset = 0; offset < dhcpcMaxLeases; offset++)
                {
                if (dhcpcLeaseList [offset] == newEvent.leaseId)
                    {
					SYSTEM_NumOfWanGet(&num_wan);
                    for (i=0; i<num_wan; i++)
                        if (pDhcpcBootCookie[i] == dhcpcLeaseList [offset])
                        {
                            pDhcpcBootCookie[i] = NULL;
                            break;
                        }

                    dhcpcLeaseList [offset] = NULL;
                    free (newEvent.leaseId);
                    break;
                    }
                }
                if (newEvent.source == DHCP_USER_EVENT &&
                        newEvent.type == DHCP_USER_RELEASE)
                    semGive(dhcpc_SemId);
            }
        }

    /* The monitor task only returns in response to a dhcpcShutdown() call. */
    semGive(dhcpc_SemId);
    return;
    }

/*******************************************************************************
*
* dhcpcEventGet - wait for a DHCP event
*
* This routine retrieves DHCP events for processing by the monitor task.
* If the contents of the event descriptor are valid, it is stored in the
* buffer provided by the <pNewEvent> parameter. Automatic event descriptors
* are created when a DHCP message arrives or a timeout expires. Manual
* event descriptors are generated by calls to the dhcpBind(), dhcpVerify(),
* dhcpRelease() and dhcpShutdown() routines.
*
* RETURNS: OK if valid event retrieved, or ERROR otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

LOCAL STATUS dhcpcEventGet
    (
    EVENT_DATA * 	pNewEvent 	/* pointer to event descriptor */
    )
    {
    STATUS 	result;
    int 	status;
    int 	offset;
    LEASE_DATA * 	pLeaseData;

    /* Wait for event occurrence. */

    semTake (dhcpcEventSem, WAIT_FOREVER);

    /* Retrieve event from message ring. */

    if (rngIsEmpty (dhcpcEventRing) == TRUE)
        {
#ifdef DHCPC_DEBUG
        logMsg ("dhcpcEventGet: Notification empty.\n", 0, 0, 0, 0, 0, 0);
#endif
        return (ERROR);
        }

    status = rngBufGet (dhcpcEventRing, (char *)pNewEvent, sizeof (EVENT_DATA));

    if (status != sizeof (EVENT_DATA))
        {
#ifdef DHCPC_DEBUG
        logMsg ("dhcpcEventGet: Notification error.\n", 0, 0, 0, 0, 0, 0);
#endif
        return (ERROR);
        }

    /*
     * Check event descriptor for valid data. The primary purpose of
     * this code is to catalog all the possible event descriptors
     * in one location, since invalid data could only result from a failure
     * in memory-to-memory copies by the ring buffers, which is extremely
     * unlikely. Even if that occurred, the DHCP protocol is designed to
     * handle any errors in message arrival notifications as a byproduct of
     * UDP processing, but corruption of timeout or user events could result
     * in unrecoverable errors.
     */

    result = OK;
    switch (pNewEvent->source)
        {
        case DHCP_AUTO_EVENT:   /* Validate automatic event types. */
            if (pNewEvent->type != DHCP_TIMEOUT &&
                    pNewEvent->type != DHCP_MSG_ARRIVED)
                {
                result = ERROR;
                }
            break;
        case DHCP_USER_EVENT:    /* Validate manual event types. */
            if (pNewEvent->type != DHCP_USER_BIND &&
                    pNewEvent->type != DHCP_USER_VERIFY &&
                    pNewEvent->type != DHCP_USER_RELEASE &&
                    pNewEvent->type != DHCP_USER_SHUTDOWN)
                {
                result = ERROR;
                }
            break;
        default:    /* Unknown event class. */
            result = ERROR;
            break;
        }

    if (result == ERROR)
        return (ERROR);

    /* Lease identifiers must be checked for all events except shutdown. */

    if (pNewEvent->source == DHCP_USER_EVENT &&
            pNewEvent->type == DHCP_USER_SHUTDOWN)
        return (OK);

    /*
     * Remove any message from the message ring, regardless of lease ID.
     * Store in receive buffer at offset to provide 4-byte alignment of
     * IP header needed by Sun BSP's.
     */

    if (pNewEvent->source == DHCP_AUTO_EVENT &&
            pNewEvent->type == DHCP_MSG_ARRIVED)
        {
        /*
         * Ignore events with bad messages - the protocol will eventually
         * timeout if no valid message arrives.
         */

        status = rngBufGet (dhcpcMsgRing, &pDhcpcRecvBuf [DHCPC_OFF],
                            pNewEvent->length);
        if (status != pNewEvent->length)
            return (ERROR);
        }

    /*
     * Although not likely, a lease could be released between the
     * arrival of an event and the event processing. In that case,
     * the recorded lease identifier will be invalid. Ignore those events.
     */

    pLeaseData = pNewEvent->leaseId;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        return (ERROR);

    return (result);
    }

/*******************************************************************************
*
* dhcpcEventHandle - process a DHCP event for a particular lease
*
* This routine executes a portion of the DHCP client finite state machine
* until a DHCP message is sent and/or the relevant timeouts are set. It
* processes all incoming DHCP messages, timeouts, and all user requests
* except for the dhcpcShutdown() routine. All handled events are processed
* in the context of the current state of a known lease. It is invoked by the
* monitor task when the events occur, and should only be called internally.
*
* RETURNS: OK if processing completed, or ERROR if it fails.
*
* ERRNO: N/A
*
* NOMANUAL
*/

STATUS dhcpcEventHandle
    (
    EVENT_DATA * 	pNewEvent 	/* pointer to event descriptor */
    )
    {
    STATUS 		result;
    LEASE_DATA * 	pLeaseData;

    /*
     * Use the cookie to access the lease-specific data structures. For now,
     * just typecast the cookie. This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pNewEvent->leaseId;

    /*
     * Execute routines from the state machine until processing is complete.
     * In general, all processing is performed by a single routine. If
     * additional routines are needed, the called routine returns DHCPC_MORE,
     * and the next routine is executed immediately. Processing always stops
     * once the occurrence of a subsequent timeout or message arrival is
     * guaranteed. The processing routine returns OK at that point, allowing
     * the monitor task to handle the next event. If a routine returns
     * ERROR, the state machine for the lease is reset to the initial state.
     * Finally, a routine returns DHCPC_DONE after releasing its lease and
     * removing the data structures.
     */

    result = DHCPC_MORE;
    while (result == DHCPC_MORE)
        {
        result = (*fsm [pLeaseData->currState]) (pNewEvent);

        if (result == ERROR)
            {
#ifdef DHCPC_DEBUG
            logMsg ("Error in finite state machine.\n", 0, 0, 0, 0, 0, 0);
#endif
            /* Lease negotiation failed - set to execute init() routine. */

            pLeaseData->prevState = DHCPC_ERROR;
            pLeaseData->currState = INIT;

            /* Disable the underlying network interface if necessary. */

            if (pLeaseData->autoConfig ||
                    pLeaseData->leaseType == DHCP_AUTOMATIC)
                {
                down_if (&pLeaseData->ifData);
                }

            /*
             * Signal the (failed) completion of the negotiation process
             * if the dhcpcBind() call is executing synchronously.
             */

            if (pLeaseData->waitFlag)
                semGive (pLeaseData->leaseSem);

            /* Send a notification of the failure to any event hook routine. */

            if (pLeaseData->eventHookRtn != NULL)
                result = (* pLeaseData->eventHookRtn) (DHCPC_LEASE_INVALID,
                                                       pNewEvent->leaseId);
            return (ERROR);
            }

        /*
         * When a lease entry is removed, return an indicator to the monitor
         * task so that is will reset the corresponding list entry to NULL.
         */

        if (result == DHCPC_DONE)
            return (DHCPC_DONE);

        if (pLeaseData->currState == BOUND && pLeaseData->prevState != BOUND)
            {
            /*
             * Signal the successful completion of the negotiation
             * process if the dhcpBind() call is executing synchronously.
             */

            if (pLeaseData->waitFlag)
                semGive (pLeaseData->leaseSem);

            /* Set the bound routine to exit after starting the timer. */

            pNewEvent->type = DHCPC_STATE_BEGIN;
            }
#ifdef DHCPC_DEBUG
        logMsg ("Next state= %d\n", pLeaseData->currState, 0, 0, 0, 0, 0);
#endif
        }
    return (OK);
    }

/*******************************************************************************
*
* dhcpcOptionFind - find the requested option
*
* This routine returns the address and length of a requested option
* for use by dhcpcOptionGet(). It should only be called internally.
*
* RETURNS: Pointer to start of data for option, or NULL if not present.
*
* ERRNO: N/A
*
* NOMANUAL
*
* INTERNAL
* The <pAmount> parameter is dereferenced unconditionally, but since this
* routine is only called internally, it is never NULL.
*/

char *dhcpcOptionFind
    (
    LEASE_DATA * 	pLeaseData, 	/* lease-specific data structures */
    int 		option,		/* RFC 1533 option tag */
    int *		pAmount		/* Number of bytes for option */
    )
    {
    char *	pData = NULL;	/* Location of option data. */
    struct dhcp_param * 	pDhcpcParam;

    pDhcpcParam = pLeaseData->dhcpcParam;

    switch (option)
        {
        case _DHCP_SUBNET_MASK_TAG:
            if (pDhcpcParam->subnet_mask != NULL)
                {
                *pAmount = sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->subnet_mask;
                }
            break;

        case _DHCP_TIME_OFFSET_TAG:
            *pAmount = sizeof (long);
            pData = (char *)&pDhcpcParam->time_offset;
            break;

        case _DHCP_ROUTER_TAG:
            if (pDhcpcParam->router != NULL)
                {
                *pAmount = pDhcpcParam->router->num * sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->router->addr;
                }
            break;

        case _DHCP_TIME_SERVER_TAG:
            if (pDhcpcParam->time_server != NULL)
                {
                *pAmount = pDhcpcParam->time_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->time_server->addr;
                }
            break;

        case _DHCP_NAME_SERVER_TAG:
            if (pDhcpcParam->name_server != NULL)
                {
                *pAmount = pDhcpcParam->name_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->name_server->addr;
                }
            break;

        case _DHCP_DNS_SERVER_TAG:
            if (pDhcpcParam->dns_server != NULL)
                {
                *pAmount = pDhcpcParam->dns_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->dns_server->addr;
                }
            break;

        case _DHCP_LOG_SERVER_TAG:
            if (pDhcpcParam->log_server != NULL)
                {
                *pAmount = pDhcpcParam->log_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->log_server->addr;
                }
            break;

        case _DHCP_COOKIE_SERVER_TAG:
            if (pDhcpcParam->cookie_server != NULL)
                {
                *pAmount = pDhcpcParam->cookie_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->cookie_server->addr;
                }
            break;

        case _DHCP_LPR_SERVER_TAG:
            if (pDhcpcParam->lpr_server != NULL)
                {
                *pAmount = pDhcpcParam->lpr_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->lpr_server->addr;
                }
            break;

        case _DHCP_IMPRESS_SERVER_TAG:
            if (pDhcpcParam->impress_server != NULL)
                {
                *pAmount = pDhcpcParam->impress_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->impress_server->addr;
                }
            break;

        case _DHCP_RLS_SERVER_TAG:
            if (pDhcpcParam->rls_server != NULL)
                {
                *pAmount = pDhcpcParam->rls_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->rls_server->addr;
                }
            break;

        case _DHCP_HOSTNAME_TAG:
            *pAmount = strlen (pDhcpcParam->hostname);
            pData = pDhcpcParam->hostname;
            break;

        case _DHCP_BOOTSIZE_TAG:
            *pAmount = sizeof (unsigned short);
            pData = (char *)&pDhcpcParam->bootsize;
            break;

        case _DHCP_MERIT_DUMP_TAG:
            *pAmount = strlen (pDhcpcParam->merit_dump);
            pData = pDhcpcParam->merit_dump;
            break;

        case _DHCP_DNS_DOMAIN_TAG:
            *pAmount = strlen (pDhcpcParam->dns_domain);
            pData = pDhcpcParam->dns_domain;
            break;

        case _DHCP_SWAP_SERVER_TAG:
            if (pDhcpcParam->swap_server != NULL)
                {
                *pAmount = sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->swap_server;
                }
            break;

        case _DHCP_ROOT_PATH_TAG:
            *pAmount = strlen (pDhcpcParam->root_path);
            pData = pDhcpcParam->root_path;
            break;

        case _DHCP_EXTENSIONS_PATH_TAG:
            *pAmount = strlen (pDhcpcParam->extensions_path);
            pData = pDhcpcParam->extensions_path;
            break;

        case _DHCP_IP_FORWARD_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->ip_forward;
            break;

        case _DHCP_NONLOCAL_SRCROUTE_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->nonlocal_srcroute;
            break;

        case _DHCP_POLICY_FILTER_TAG:
            if (pDhcpcParam->policy_filter != NULL)
                {
                *pAmount = pDhcpcParam->policy_filter->num *
                           2 * sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->policy_filter->addr;
                }
            break;

        case _DHCP_MAX_DGRAM_SIZE_TAG:
            *pAmount = sizeof (unsigned short);
            pData = (char *)&pDhcpcParam->max_dgram_size;
            break;

        case _DHCP_DEFAULT_IP_TTL_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->default_ip_ttl;
            break;

        case _DHCP_MTU_AGING_TIMEOUT_TAG:
            *pAmount = sizeof (unsigned long);
            pData = (char *)&pDhcpcParam->mtu_aging_timeout;
            break;

        case _DHCP_MTU_PLATEAU_TABLE_TAG:
            if (pDhcpcParam->mtu_plateau_table != NULL)
                {
                *pAmount = pDhcpcParam->mtu_plateau_table->num *
                           sizeof (unsigned short);
                pData = (char *)pDhcpcParam->mtu_plateau_table->shortnum;
                }
            break;

        case _DHCP_IF_MTU_TAG:
            *pAmount = sizeof (unsigned short);
            pData = (char *)&pDhcpcParam->intf_mtu;
            break;

        case _DHCP_ALL_SUBNET_LOCAL_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->all_subnet_local;
            break;

        case _DHCP_BRDCAST_ADDR_TAG:
            if (pDhcpcParam->brdcast_addr != NULL)
                {
                *pAmount = sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->brdcast_addr;
                }
            break;

        case _DHCP_MASK_DISCOVER_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->mask_discover;
            break;

        case _DHCP_MASK_SUPPLIER_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->mask_supplier;
            break;

        case _DHCP_ROUTER_DISCOVER_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->router_discover;
            break;

        case _DHCP_ROUTER_SOLICIT_TAG:
            if (pDhcpcParam->router_solicit.s_addr != 0)
                {
                *pAmount = sizeof (struct in_addr);
                pData = (char *)&pDhcpcParam->router_solicit;
                }
            break;

        case _DHCP_STATIC_ROUTE_TAG:
            if (pDhcpcParam->static_route != NULL)
                {
                *pAmount = pDhcpcParam->static_route->num *
                           2 * sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->static_route->addr;
                }
            break;

        case _DHCP_TRAILER_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->trailer;
            break;

        case _DHCP_ARP_CACHE_TIMEOUT_TAG:
            *pAmount = sizeof (unsigned long);
            pData = (char *)&pDhcpcParam->arp_cache_timeout;
            break;

        case _DHCP_ETHER_ENCAP_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->ether_encap;
            break;

        case _DHCP_DEFAULT_TCP_TTL_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->default_tcp_ttl;
            break;

        case _DHCP_KEEPALIVE_INTERVAL_TAG:
            *pAmount = sizeof (unsigned long);
            pData = (char *)&pDhcpcParam->keepalive_inter;
            break;

        case _DHCP_KEEPALIVE_GARBAGE_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->keepalive_garba;
            break;

        case _DHCP_NIS_DOMAIN_TAG:
            *pAmount = strlen (pDhcpcParam->nis_domain);
            pData = pDhcpcParam->nis_domain;
            break;

        case _DHCP_NIS_SERVER_TAG:
            if (pDhcpcParam->nis_server != NULL)
                {
                *pAmount = pDhcpcParam->nis_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->nis_server->addr;
                }
            break;

        case _DHCP_NTP_SERVER_TAG:
            if (pDhcpcParam->ntp_server != NULL)
                {
                *pAmount = pDhcpcParam->ntp_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->ntp_server->addr;
                }
            break;

        case _DHCP_NBN_SERVER_TAG:
            if (pDhcpcParam->nbn_server != NULL)
                {
                *pAmount = pDhcpcParam->nbn_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->nbn_server->addr;
                }
            break;

        case _DHCP_NBDD_SERVER_TAG:
            if (pDhcpcParam->nbdd_server != NULL)
                {
                *pAmount = pDhcpcParam->nbdd_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->nbdd_server->addr;
                }
            break;

        case _DHCP_NB_NODETYPE_TAG:
            *pAmount = sizeof (unsigned char);
            pData = (char *)&pDhcpcParam->nb_nodetype;
            break;

        case _DHCP_NB_SCOPE_TAG:
            *pAmount = strlen (pDhcpcParam->nb_scope);
            pData = pDhcpcParam->nb_scope;
            break;

        case _DHCP_XFONT_SERVER_TAG:
            if (pDhcpcParam->xfont_server != NULL)
                {
                *pAmount = pDhcpcParam->xfont_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->xfont_server->addr;
                }
            break;

        case _DHCP_XDISPLAY_MANAGER_TAG:
            if (pDhcpcParam->xdisplay_manager != NULL)
                {
                *pAmount = pDhcpcParam->xdisplay_manager->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->xdisplay_manager->addr;
                }
            break;

        case _DHCP_LEASE_TIME_TAG:
            *pAmount = sizeof (unsigned long);
            pData = (char *)&pDhcpcParam->lease_duration;
            break;

        case _DHCP_SERVER_ID_TAG:
            *pAmount = sizeof (struct in_addr);
            pData = (char *)&pDhcpcParam->server_id;
            break;

        case _DHCP_ERRMSG_TAG:
            *pAmount = strlen (pDhcpcParam->errmsg);
            pData = pDhcpcParam->errmsg;
            break;

        case _DHCP_T1_TAG:
            *pAmount = sizeof (unsigned long);
            pData = (char *)&pDhcpcParam->dhcp_t1;
            break;

        case _DHCP_T2_TAG:
            *pAmount = sizeof (unsigned long);
            pData = (char *)&pDhcpcParam->dhcp_t2;
            break;

        case _DHCP_NISP_DOMAIN_TAG:
            *pAmount = strlen (pDhcpcParam->nisp_domain);
            pData = pDhcpcParam->nisp_domain;
            break;

        case _DHCP_NISP_SERVER_TAG:
            if (pDhcpcParam->nisp_server != NULL)
                {
                *pAmount = pDhcpcParam->nisp_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->nisp_server->addr;
                }
            break;

        case _DHCP_MOBILEIP_HA_TAG:
            if (pDhcpcParam->mobileip_ha != NULL)
                {
                *pAmount = pDhcpcParam->mobileip_ha->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->mobileip_ha->addr;
                }
            break;

        case _DHCP_SMTP_SERVER_TAG:
            if (pDhcpcParam->smtp_server != NULL)
                {
                *pAmount = pDhcpcParam->smtp_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->smtp_server->addr;
                }
            break;

        case _DHCP_POP3_SERVER_TAG:
            if (pDhcpcParam->pop3_server != NULL)
                {
                *pAmount = pDhcpcParam->pop3_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->pop3_server->addr;
                }
            break;

        case _DHCP_NNTP_SERVER_TAG:
            if (pDhcpcParam->nntp_server != NULL)
                {
                *pAmount = pDhcpcParam->nntp_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->nntp_server->addr;
                }
            break;

        case _DHCP_DFLT_WWW_SERVER_TAG:
            if (pDhcpcParam->dflt_www_server != NULL)
                {
                *pAmount = pDhcpcParam->dflt_www_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->dflt_www_server->addr;
                }
            break;

        case _DHCP_DFLT_FINGER_SERVER_TAG:
            if (pDhcpcParam->dflt_finger_server != NULL)
                {
                *pAmount = pDhcpcParam->dflt_finger_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->dflt_finger_server->addr;
                }
            break;

        case _DHCP_DFLT_IRC_SERVER_TAG:
            if (pDhcpcParam->dflt_irc_server != NULL)
                {
                *pAmount = pDhcpcParam->dflt_irc_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->dflt_irc_server->addr;
                }
            break;

        case _DHCP_STREETTALK_SERVER_TAG:
            if (pDhcpcParam->streettalk_server != NULL)
                {
                *pAmount = pDhcpcParam->streettalk_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->streettalk_server->addr;
                }
            break;

        case _DHCP_STDA_SERVER_TAG:
            if (pDhcpcParam->stda_server != NULL)
                {
                *pAmount = pDhcpcParam->stda_server->num *
                           sizeof (struct in_addr);
                pData = (char *)pDhcpcParam->stda_server->addr;
                }
            break;

        default:
            break;
        }
    return (pData);
    }

/*******************************************************************************
*
* dhcpcDestCheck - prepare incoming messages for further examination
*
* This routine checks incoming network traffic for UDP packets arriving
* at the DHCP client port. Any Ethernet frames which match those criteria
* are copied to an interface-specific buffer for further testing. This
* routine is called by the internal packet filter as the first step in
* retrieving DHCP server replies.
*
* RETURNS: OK if tests pass, or ERROR if message doesn't match expected values.
*
* ERRNO: N/A
*
* NOMANUAL
*/

LOCAL STATUS dhcpcDestCheck
    (
    char * 	pInput, 	/* contents of received packet */
    int 	length 		/* length of received packet */
    )
    {
    struct ether_header * 	pEtherHdr;    /* pointer to ethernet header */
    struct ip * 		pIpHdr;    /* pointer to IP header */
    struct udphdr * 		pUdpHdr;   /* pointer to UDP header */

    pEtherHdr = (struct ether_header *) pInput;
    pIpHdr = (struct ip *) &pInput [SIZEOF_ETHERHEADER];
#if BSD<44
    pUdpHdr = (struct udphdr *) &pInput [SIZEOF_ETHERHEADER +
                                         (pIpHdr->ip_v_hl & 0xf) *
                                         _BYTESPERWORD];
#else
    pUdpHdr = (struct udphdr *) &pInput [SIZEOF_ETHERHEADER + pIpHdr->ip_hl *
                                         _BYTESPERWORD];
#endif

    if (length <= SIZEOF_ETHERHEADER)
        return (ERROR);

    if (ntohs (pEtherHdr->ether_type) != ETHERTYPE_IP)
        return (ERROR);

    if (pIpHdr->ip_p != IPPROTO_UDP)
        return (ERROR);

    if (pUdpHdr->uh_dport != dhcpc_port)
        return (ERROR);

    /*
     * Copy the input packet one byte at a time to accomodate board
     * specific memory access requirements. Use offset to provide
     * 4-byte alignment of IP header needed by Sun BSP's.
     */

    bcopyBytes (pInput, &dhcpif.rbuf [DHCPC_OFF], length);

    /* WIDE project DHCP code: initializes pointers. */
    align_msg (&dhcpcIfMsg, &dhcpif.rbuf [DHCPC_OFF]);

    return (OK);
    }


/*******************************************************************************
*
* dhcpcXidVerify - verify that the transaction ID belongs to a known lease
*
* This routine compares the transaction identifier in an incoming DHCP
* message with all known transaction identifiers in use by the available
* leases. If no match is found, the corresponding DHCP message will be
* rejected.
*
* RETURNS: Lease identifier if transaction ID recognized, or NULL otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

void * dhcpcXidVerify
    (
    unsigned long transId
    )
    {
    int loop;
    void * 	pLeaseId = NULL;

    for (loop = 0; loop < dhcpcMaxLeases; loop++)
	{
        if (dhcpcLeaseList [loop] != NULL &&
                transId == dhcpcLeaseList [loop]->xid)
            {
            pLeaseId = dhcpcLeaseList [loop];
            break;
            }
	}
    if (loop == dhcpcMaxLeases)
        return (NULL);

    return (pLeaseId);
    }

/*******************************************************************************
*
* dhcpcBodyCheck - verify that a received packet is a valid server reply
*
* This routine examines an IP message received at the DHCP client port to
* determine if it is a DHCP or BOOTP server reply. It is used internally by
* the Ethernet input hook as the final step in detecting incoming DHCP/BOOTP
* messages.
*
* RETURNS: OK if message recognized, or ERROR otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

STATUS dhcpcBodyCheck (void)
    {
    u_short ripcksum = 0;
    u_short rudpcksum = 0;
    struct ps_udph rpudph;

    /* Check if UDP and DHCP portions are present (set by align_msg). */

    if (dhcpcIfMsg.udp == NULL || dhcpcIfMsg.dhcp == NULL)
        return (ERROR);

    /* Construct a UDP pseudo-header. */

    bzero ( (char *)&rpudph, sizeof (rpudph));

    ripcksum = dhcpcIfMsg.ip->ip_sum;
    dhcpcIfMsg.ip->ip_sum = 0;
    rudpcksum = dhcpcIfMsg.udp->uh_sum;
    dhcpcIfMsg.udp->uh_sum = 0;
    rpudph.zero = 0;
    rpudph.prto = IPPROTO_UDP;
    rpudph.srcip.s_addr = dhcpcIfMsg.ip->ip_src.s_addr;
    rpudph.dstip.s_addr = dhcpcIfMsg.ip->ip_dst.s_addr;
    rpudph.ulen = dhcpcIfMsg.udp->uh_ulen;

    /* Check lengths of each portion of message. */

    if (ntohs (dhcpcIfMsg.ip->ip_len) < DFLTBOOTPLEN + UDPHL + IPHL)
        return (ERROR);

    if (ntohs (dhcpcIfMsg.udp->uh_ulen) < DFLTBOOTPLEN + UDPHL)
        return (ERROR);

    /* Verify message received at client port. */

    if (dhcpcIfMsg.udp->uh_dport != dhcpc_port)
        return (ERROR);

    /* Check message IP checksum. */

#if BSD<44
    if (ripcksum != checksum ( (u_short *)dhcpcIfMsg.ip,
                               (dhcpcIfMsg.ip->ip_v_hl & 0xf) << 2))
#else
    if (ripcksum != checksum ( (u_short *)dhcpcIfMsg.ip,
                               dhcpcIfMsg.ip->ip_hl << 2))
#endif
        return (ERROR);

    /* Check message UDP checksum. */

    if (dhcpcIfMsg.udp->uh_sum != 0 &&
        rudpcksum != udp_cksum (&rpudph, (char *)dhcpcIfMsg.udp,
                                ntohs (rpudph.ulen)))
        return (ERROR);

    /* Check DHCP message opcode, transaction ID, and magic cookie. */

    if (dhcpcIfMsg.dhcp->op != BOOTREPLY)
        return (ERROR);

    if (bcmp (dhcpcIfMsg.dhcp->options, (char *)dhcpCookie, MAGIC_LEN) != 0)
        return (ERROR);

    return (OK);
    }

/******************************************************************************
*
* dhcpcDefaultsSet - assign host requirements defaults for client
*
* This routine fills the client's parameters structure with the default
* values specified in the Host Requirements Documents (RFC's 1122 and 1123),
* the Path MTU Discovery description (RFC 1191), or the Router Discovery
* specification (RFC 1256). This data is assigned before processing a message
* received from a DHCP server, so that it can override the defaults.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void dhcpcDefaultsSet
    (
    struct dhcp_param * 	pNewParam
    )
    {
    /* Default IP layer parameters, per host. */

    pNewParam->ip_forward = FALSE;
    pNewParam->nonlocal_srcroute = FALSE;
    pNewParam->max_dgram_size = _HRD_MAX_DGRAM;
    pNewParam->default_ip_ttl = _HRD_IP_TTL;

    /* Default IP layer parameters, per interface. */

    pNewParam->intf_mtu = _HRD_MTU;
    pNewParam->all_subnet_local = FALSE;
    pNewParam->mask_discover = FALSE;
    pNewParam->mask_supplier = FALSE;
    pNewParam->router_discover = TRUE;
    pNewParam->router_solicit.s_addr = _HRD_ROUTER;

    /* Default link layer parameters, per interface. */

    pNewParam->trailer = FALSE;
    pNewParam->arp_cache_timeout = _HRD_ARP_TIME;
    pNewParam->ether_encap = FALSE;

    /* Default link layer parameters, per host. */

    pNewParam->default_tcp_ttl = _HRD_TCP_TTL;
    pNewParam->keepalive_inter = _HRD_TCP_TIME;
    pNewParam->keepalive_garba = FALSE;
    }

/******************************************************************************
*
* dhcpcEventAdd - send event notification to monitor task
*
* This routine adds event descriptors to the event queue for later handling
* by the DHCP client monitor task.
*
* If the <source> parameter is set to DHCP_USER_EVENT, the routine was called
* by one of the following API routines: dhcpcBind(), dhcpcRelease(),
* dhcpcVerify(), or dhcpcShutdown(). The <type> parameter indicates the
* corresponding action: DHCP_USER_BIND to initiate the lease negotiation
* process, DHCP_USER_VERIFY to verify an active lease, DHCP_USER_RELEASE to
* relinquish an active lease, and DHCP_USER_SHUTDOWN to release all active
* leases and disable the DHCP client library.
*
* If the <source> parameter is set to DHCP_AUTO_EVENT, the routine was called
* in response to a timeout in the DHCP client state machine and the
* <type> parameter is set to DHCP_TIMEOUT.
*
* The <pLeaseId> parameter identifies the associated lease for timeout events
* and the first three user events.
*
* The <intFlag> parameter indicates if this routine is executing at interrupt
* level. If it is FALSE, then interrupts must be locked out to prevent
* write conflicts to the event ring buffer between user requests and watchdog
* timers.
*
* RETURNS: OK if event added successfully, or ERROR otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

STATUS dhcpcEventAdd
    (
    int 	source, 	/* automatic event or user request */
    int 	type, 		/* event identifier */
    void * 	pLeaseId,	/* internal lease identifier */
    BOOL 	intFlag 	/* executing at interrupt level? */
    )
    {
    EVENT_DATA 	newEvent;
    int 	status;
    int 	key = 0;

    newEvent.source = source;
    newEvent.type = type;
    newEvent.leaseId = pLeaseId;

    /*
     * Add the event to the monitor task's event list.
     * Disable interrupts if necessary.
     */

    if (!intFlag)
        key = intLock ();
    status = rngBufPut (dhcpcEventRing, (char *)&newEvent, sizeof (newEvent));
    if (!intFlag)
        intUnlock (key);

    if (status != sizeof (newEvent))
        return (ERROR);

    /* Signal the monitor task of the event arrival. */

    status = semGive (dhcpcEventSem);
    if (status == ERROR)
        return (ERROR);

    return (OK);
    }

/******************************************************************************
*
* dhcpcInputHook - generic packet filter for DHCP messages
*
* This routine extracts DHCP server responses from incoming ethernet traffic.
* It is called by the network interface driver when a new input frame comes in
* from the network. It is attached to the network driver with etherHook
* routines.
*
* RETURNS: TRUE if the ethernet frame is handled by this routine. No further
* processing by the network driver is needed.
* FALSE if the frame is not a server response. The ethernet frame is
* then handled by the network driver.
*
* ERRNO: N/A
*
* NOMANUAL
*/

BOOL dhcpcInputHook
    (
    struct ifnet *	pIf,        /* interface where packet was received */
    FAST char *		pInput,     /* contents of received packet */
    FAST int 		length      /* length of received packet */
    )
    {
    int retVal;                    /* Return values */
    int key;
    EVENT_DATA newEvent;
    void * 	pLeaseId;

    /* Adjust message size (ignores trailers added by some drivers). */

   if (length > DHCP_MSG_SIZE)
       length = DHCP_MSG_SIZE;

    /*
     * These routines access the dhcpcIfMsg and dhcpif globals. Those
     * variables must be replaced with interface-specific equivalents.
     */

    /* Copy messages received at client port to interface-specific buffer. */

    retVal = dhcpcDestCheck (pInput, length);
    if (retVal != OK)
        return (FALSE);
    /*
     * Validate message checksums and lengths. Verify "op" field, cookie
     * and transaction identifier. This routine will accept BOOTP or DHCP
     * responses. (The tests also alter the received message - must return
     * TRUE to network driver after this point).
     */

    retVal = dhcpcBodyCheck ();
    if (retVal == OK)
        {
        /* DHCP message found - check for known transaction ID. */

        pLeaseId = dhcpcXidVerify (dhcpcIfMsg.dhcp->xid);
        if (pLeaseId == NULL)
            return (TRUE);

        /* Signal WIDE project DHCP code when a valid message arrives. */

        newEvent.source = DHCP_AUTO_EVENT;
        newEvent.type = DHCP_MSG_ARRIVED;
        newEvent.leaseId = pLeaseId;
        newEvent.length = length;

        /* Lock interrupts to prevent conflicts with timeout thread. */

        key = intLock ();
        retVal = rngBufPut (dhcpcEventRing, (char *) &newEvent,
                            sizeof(newEvent));
        intUnlock (key);

        if (retVal == sizeof (newEvent))
            {
            /* Ignore storage error - notifications with no message are OK. */

            rngBufPut (dhcpcMsgRing, pInput, length);
            semGive (dhcpcEventSem);
            }
        }
    return (TRUE);
    }


/******************************************************************************
*
* dhcpArpInputHook - packet filter for ARP replies
*
* This routine filters out ARP replies from incoming ethernet traffic.
* It is called by the network interface driver when a new input frame comes
* in from the network. It is attached to the network driver with etherHook
* routines.
*
* RETURNS: TRUE if the ethernet frame is handled by this routine. No further
*          processing by the network driver is needed.
*          FALSE if the frame is not an ARP reply. The ethernet frame is then
*          handled by the network driver.
*
* ERRNO: N/A
*
* NOMANUAL
*/

BOOL dhcpArpInputHook
    (
    struct ifnet *	pIf,        /* interface where packet was received */
    FAST char *		pInput,     /* contents of received packet */
    FAST int 		length      /* length of received packet */
    )
    {
    FAST struct ether_header *	pEtherHdr;    /* pointer to ethernet header */
    FAST struct ether_arp *	pArp; 	/* pointer to ethernet ARP packet */
    int bufLen;

    pEtherHdr = (struct ether_header *)pInput;

    if (length <= SIZEOF_ETHERHEADER)
        return (FALSE);

    if (ntohs (pEtherHdr->ether_type) != ETHERTYPE_ARP)
        return (FALSE);

    /*
     * Copy the input packet one byte at a time to accomodate board
     * specific memory access requirements.
     */
    bufLen = length - SIZEOF_ETHERHEADER;

    /* Adjust message size (ignores trailers added by some drivers). */

   if (bufLen > sizeof (struct ether_arp))
       bufLen = sizeof (struct ether_arp);

    bcopyBytes ( (char *) ( (u_char *)pInput + SIZEOF_ETHERHEADER),
                inputBuffer, bufLen);
    pArp = (struct ether_arp *)&inputBuffer;

    /*
     *  Check if the packet is a valid ARP reply.
     */

    if (ntohs (pArp->arp_op) != ARPOP_REPLY)
        return FALSE;

    /* Check address of sender against (our) address stored in global. */

    if (bcmp ( (char *)dhcpArpSpa, (char *)pArp->arp_spa, pArp->arp_pln) != 0)
        return FALSE;

    /* Valid reply: signal state machine to continue. */
    semGive (dhcpcArpEventSem);

    /* Delete input hook so only one ARP message is accepted. */

    if (muxDevExists (pIf->if_name, pIf->if_unit))
        etherInputHookDelete (dhcpArpInputHook, pIf->if_name, pIf->if_unit);
    else
        etherInputHookDelete (dhcpArpInputHook, NULL, pIf->if_unit);

    return (TRUE);
    }


/******************************************************************************
*
* dhcpcParamsCopy - copy current network parameters.
*
* This routine copies the network parameters to the caller-supplied structure
* from the lease indicated by the <pLeaseData> parameter. It is called
* internally by the dhcpcParamsGet() routine.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*
* INTERNAL
* The data structure accessed by this routine was defined by the original
* public domain code used as a basis for the DHCP implementation. The
* structure tag and field names are not required to follow the coding
* standards (and they don't).
*/

void dhcpcParamsCopy
    (
    LEASE_DATA * 	pLeaseData,
    struct dhcp_param * pParamList
    )
    {
    int loop;
    int limit = 0;
    struct dhcp_param * pDhcpcParam;

    /*
     * The DHCP client parameters received from the server are stored in a
     * large structure which either contains the actual parameter value
     * or a pointer to the data. Because the parameters received are
     * determined by the server configuration and many parameters are
     * optional, any of the fields could use the default value determined
     * earlier. In most cases, the default value is an empty field (or
     * zero in the case of numeric values). However, some parameters contain
     * default settings specified by the host requirements documents which are
     * used if the server does not provide a value. The following code copies
     * any available parameters into the provided target structure. If
     * necessary, it checks for empty fields. The code is divided into logical
     * blocks according to the various data types used by the DHCP parameters.
     */

    pDhcpcParam = pLeaseData->dhcpcParam;

    /* Process all string parameters. */

    bcopy (pDhcpcParam->got_option, pParamList->got_option, GOTOPTSIZ);

    if (pParamList->sname != NULL && pDhcpcParam->sname != NULL)
        strcpy (pParamList->sname, pDhcpcParam->sname);

    if (pParamList->file  != NULL && pDhcpcParam->file != NULL)
        strcpy (pParamList->file, pDhcpcParam->file);

    if (pParamList->hostname != NULL && pDhcpcParam->hostname != NULL)
        strcpy (pParamList->hostname, pDhcpcParam->hostname);

    if (pParamList->merit_dump != NULL && pDhcpcParam->merit_dump != NULL)
        strcpy (pParamList->merit_dump, pDhcpcParam->merit_dump);

    if (pParamList->dns_domain != NULL && pDhcpcParam->dns_domain != NULL)
        strcpy (pParamList->dns_domain, pDhcpcParam->dns_domain);

    if (pParamList->root_path != NULL && pDhcpcParam->root_path != NULL)
        strcpy (pParamList->root_path, pDhcpcParam->root_path);

    if (pParamList->extensions_path != NULL &&
        pDhcpcParam->extensions_path != NULL)
        strcpy (pParamList->extensions_path, pDhcpcParam->extensions_path);

    if (pParamList->nis_domain != NULL && pDhcpcParam->nis_domain != NULL)
        strcpy (pParamList->nis_domain, pDhcpcParam->nis_domain);

    if (pParamList->nb_scope != NULL && pDhcpcParam->nb_scope != NULL)
        strcpy (pParamList->nb_scope, pDhcpcParam->nb_scope);

    if (pParamList->errmsg != NULL && pDhcpcParam->errmsg != NULL)
        strcpy (pParamList->errmsg, pDhcpcParam->errmsg);

    if (pParamList->nisp_domain != NULL && pDhcpcParam->nisp_domain != NULL)
        strcpy (pParamList->nisp_domain, pDhcpcParam->nisp_domain);

    /* Process all boolean parameters. */

    pParamList->ip_forward = pDhcpcParam->ip_forward;
    pParamList->nonlocal_srcroute = pDhcpcParam->nonlocal_srcroute;
    pParamList->all_subnet_local = pDhcpcParam->all_subnet_local;
    pParamList->mask_discover = pDhcpcParam->mask_discover;
    pParamList->mask_supplier = pDhcpcParam->mask_supplier;
    pParamList->router_discover = pDhcpcParam->router_discover;
    pParamList->trailer = pDhcpcParam->trailer;
    pParamList->ether_encap = pDhcpcParam->ether_encap;
    pParamList->keepalive_garba = pDhcpcParam->keepalive_garba;

    /* Process all single numeric parameters. */

    pParamList->time_offset = pDhcpcParam->time_offset;
    pParamList->bootsize = pDhcpcParam->bootsize;
    pParamList->max_dgram_size = pDhcpcParam->max_dgram_size;
    pParamList->default_ip_ttl = pDhcpcParam->default_ip_ttl;
    pParamList->mtu_aging_timeout = pDhcpcParam->mtu_aging_timeout;
    pParamList->intf_mtu = pDhcpcParam->intf_mtu;
    pParamList->arp_cache_timeout = pDhcpcParam->arp_cache_timeout;
    pParamList->default_tcp_ttl = pDhcpcParam->default_tcp_ttl;
    pParamList->keepalive_inter = pDhcpcParam->keepalive_inter;
    pParamList->nb_nodetype = pDhcpcParam->nb_nodetype;
    pParamList->lease_origin = pDhcpcParam->lease_origin;
    pParamList->lease_duration = pDhcpcParam->lease_duration;
    pParamList->dhcp_t1 = pDhcpcParam->dhcp_t1;
    pParamList->dhcp_t2 = pDhcpcParam->dhcp_t2;

    /* Process multiple numeric parameters. */

    if (pParamList->mtu_plateau_table != NULL &&
        pDhcpcParam->mtu_plateau_table != NULL)
        {
        limit = 0;
        if (pParamList->mtu_plateau_table->shortnum != NULL)
            {
            limit = (pParamList->mtu_plateau_table->num <
                     pDhcpcParam->mtu_plateau_table->num) ?
                     pParamList->mtu_plateau_table->num :
                     pDhcpcParam->mtu_plateau_table->num;
            for (loop = 0; loop < limit; loop++)
                {
                pParamList->mtu_plateau_table->shortnum [loop] =
                pDhcpcParam->mtu_plateau_table->shortnum [loop];
                }
            }
        pParamList->mtu_plateau_table->num = limit;
        }

    /* Process single IP addresses. */

    pParamList->server_id = pDhcpcParam->server_id;
    pParamList->ciaddr = pDhcpcParam->ciaddr;
    pParamList->yiaddr = pDhcpcParam->yiaddr;
    pParamList->siaddr = pDhcpcParam->siaddr;
    pParamList->giaddr = pDhcpcParam->giaddr;

    if (pParamList->subnet_mask != NULL && pDhcpcParam->subnet_mask != NULL)
        bcopy ( (char *)pDhcpcParam->subnet_mask,
               (char *)pParamList->subnet_mask, sizeof (struct in_addr));

    if (pParamList->swap_server != NULL && pDhcpcParam->swap_server != NULL)
        bcopy ( (char *)pDhcpcParam->swap_server,
               (char *)pParamList->swap_server, sizeof (struct in_addr));

    if (pParamList->brdcast_addr != NULL &&
        pDhcpcParam->brdcast_addr != NULL)
        bcopy ( (char *)pDhcpcParam->brdcast_addr,
               (char *)pParamList->brdcast_addr, sizeof (struct in_addr));

    pParamList->router_solicit = pDhcpcParam->router_solicit;

    /* Process multiple IP addresses. */

    if (pParamList->router != NULL && pDhcpcParam->router != NULL)
        {
        limit = 0;
        if (pParamList->router->addr != NULL)
            {
            limit = (pParamList->router->num < pDhcpcParam->router->num) ?
                     pParamList->router->num : pDhcpcParam->router->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->router->addr [loop],
                       (char *)&pParamList->router->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->router->num = limit;
        }

    if (pParamList->time_server != NULL && pDhcpcParam->time_server != NULL)
        {
        limit = 0;
        if (pParamList->time_server->addr != NULL)
            {
            limit = (pParamList->time_server->num <
                     pDhcpcParam->time_server->num) ?
                     pParamList->time_server->num :
                     pDhcpcParam->time_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->time_server->addr [loop],
                       (char *)&pParamList->time_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->time_server->num = limit;
        }

    if (pParamList->name_server != NULL && pDhcpcParam->name_server != NULL)
        {
        limit = 0;
        if (pParamList->name_server->addr != NULL)
            {
            limit = (pParamList->name_server->num <
                     pDhcpcParam->name_server->num) ?
                     pParamList->name_server->num :
                     pDhcpcParam->name_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->name_server->addr [loop],
                       (char *)&pParamList->name_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->name_server->num = limit;
        }

    if (pParamList->dns_server != NULL && pDhcpcParam->dns_server != NULL)
        {
        limit = 0;
        if (pParamList->dns_server->addr != NULL)
            {
            limit = (pParamList->dns_server->num <
                     pDhcpcParam->dns_server->num) ?
                     pParamList->dns_server->num :
                     pDhcpcParam->dns_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->dns_server->addr [loop],
                       (char *)&pParamList->dns_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->dns_server->num = limit;
        }

    if (pParamList->log_server != NULL && pDhcpcParam->log_server != NULL)
        {
        limit = 0;
        if (pParamList->log_server->addr != NULL)
            {
            limit = (pParamList->log_server->num <
                     pDhcpcParam->log_server->num) ?
                     pParamList->log_server->num :
                     pDhcpcParam->log_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->log_server->addr [loop],
                       (char *)&pParamList->log_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->log_server->num = limit;
        }

    if (pParamList->cookie_server != NULL &&
        pDhcpcParam->cookie_server != NULL)
        {
        limit = 0;
        if (pParamList->cookie_server->addr != NULL)
            {
            limit = (pParamList->log_server->num <
                     pDhcpcParam->log_server->num) ?
                     pParamList->log_server->num :
                     pDhcpcParam->log_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->cookie_server->addr [loop],
                       (char *)&pParamList->cookie_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->cookie_server->num = limit;
        }

    if (pParamList->lpr_server != NULL && pDhcpcParam->lpr_server != NULL)
        {
        limit = 0;
        if (pParamList->lpr_server->addr != NULL)
            {
            limit = (pParamList->lpr_server->num <
                     pDhcpcParam->lpr_server->num) ?
                     pParamList->lpr_server->num :
                     pDhcpcParam->lpr_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy( (char *)&pDhcpcParam->lpr_server->addr [loop],
                      (char *)&pParamList->lpr_server->addr [loop],
                      sizeof (struct in_addr));
                }
            }
        pParamList->lpr_server->num = limit;
        }

    if (pParamList->impress_server != NULL &&
        pDhcpcParam->impress_server != NULL)
        {
        limit = 0;
        if (pParamList->impress_server->addr != NULL)
            {
            limit = (pParamList->impress_server->num <
                     pDhcpcParam->impress_server->num) ?
                     pParamList->impress_server->num :
                     pDhcpcParam->impress_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->impress_server->addr [loop],
                       (char *)&pParamList->impress_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->impress_server->num = limit;
        }

    if (pParamList->rls_server != NULL && pDhcpcParam->rls_server != NULL)
        {
        limit = 0;
        if (pParamList->rls_server->addr != NULL)
            {
            limit = (pParamList->rls_server->num <
                     pDhcpcParam->rls_server->num) ?
                     pParamList->rls_server->num :
                     pDhcpcParam->rls_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->rls_server->addr [loop],
                       (char *)&pParamList->rls_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->rls_server->num = limit;
        }

    if (pParamList->nis_server != NULL && pDhcpcParam->nis_server != NULL)
        {
        limit = 0;
        if (pParamList->nis_server->addr != NULL)
            {
            limit = (pParamList->nis_server->num <
                     pDhcpcParam->nis_server->num) ?
                     pParamList->nis_server->num :
                     pDhcpcParam->nis_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->nis_server->addr [loop],
                       (char *)&pParamList->nis_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->nis_server->num = limit;
        }

    if (pParamList->ntp_server != NULL && pDhcpcParam->ntp_server != NULL)
        {
        limit = 0;
        if (pParamList->ntp_server->addr != NULL)
            {
            limit = (pParamList->ntp_server->num <
                     pDhcpcParam->ntp_server->num) ?
                     pParamList->ntp_server->num :
                     pDhcpcParam->ntp_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->ntp_server->addr [loop],
                       (char *)&pParamList->ntp_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->ntp_server->num = limit;
        }

    if (pParamList->nbn_server != NULL && pDhcpcParam->nbn_server != NULL)
        {
        limit = 0;
        if (pParamList->nbn_server->addr != NULL)
            {
            limit = (pParamList->nbn_server->num <
                     pDhcpcParam->nbn_server->num) ?
                     pParamList->nbn_server->num :
                     pDhcpcParam->nbn_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->nbn_server->addr [loop],
                       (char *)&pParamList->nbn_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->nbn_server->num = limit;
        }

    if (pParamList->nbdd_server != NULL && pDhcpcParam->nbdd_server != NULL)
        {
        limit = 0;
        if (pParamList->nbdd_server->addr != NULL)
            {
            limit = (pParamList->nbdd_server->num <
                     pDhcpcParam->nbdd_server->num) ?
                     pParamList->nbdd_server->num :
                     pDhcpcParam->nbdd_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->nbdd_server->addr [loop],
                       (char *)&pParamList->nbdd_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->nbdd_server->num = limit;
        }

    if (pParamList->xfont_server != NULL &&
        pDhcpcParam->xfont_server != NULL)
        {
        limit = 0;
        if (pParamList->xfont_server->addr != NULL)
            {
            limit = (pParamList->xfont_server->num <
                     pDhcpcParam->xfont_server->num) ?
                     pParamList->xfont_server->num :
                     pDhcpcParam->xfont_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->xfont_server->addr [loop],
                       (char *)&pParamList->xfont_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->xfont_server->num = limit;
        }

    if (pParamList->xdisplay_manager != NULL &&
        pDhcpcParam->xdisplay_manager != NULL)
        {
        limit = 0;
        if (pParamList->xdisplay_manager->addr != NULL)
            {
            limit = (pParamList->xdisplay_manager->num <
                     pDhcpcParam->xdisplay_manager->num) ?
                     pParamList->xdisplay_manager->num :
                     pDhcpcParam->xdisplay_manager->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->xdisplay_manager->addr [loop],
                       (char *)&pParamList->xdisplay_manager->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->xdisplay_manager->num = limit;
        }

    if (pParamList->nisp_server != NULL && pDhcpcParam->nisp_server != NULL)
        {
        limit = 0;
        if (pParamList->nisp_server->addr != NULL)
            {
            limit = (pParamList->nisp_server->num <
                     pDhcpcParam->nisp_server->num) ?
                     pParamList->nisp_server->num :
                     pDhcpcParam->nisp_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->nisp_server->addr [loop],
                       (char *)&pParamList->nisp_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->nisp_server->num = limit;
        }

    if (pParamList->mobileip_ha != NULL && pDhcpcParam->mobileip_ha != NULL)
        {
        limit = 0;
        if (pParamList->mobileip_ha->addr != NULL)
            {
            limit = (pParamList->mobileip_ha->num <
                     pDhcpcParam->mobileip_ha->num) ?
                     pParamList->mobileip_ha->num :
                     pDhcpcParam->mobileip_ha->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->mobileip_ha->addr [loop],
                       (char *)&pParamList->mobileip_ha->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->mobileip_ha->num = limit;
        }

    if (pParamList->smtp_server != NULL && pDhcpcParam->smtp_server != NULL)
        {
        limit = 0;
        if (pParamList->smtp_server->addr != NULL)
            {
            limit = (pParamList->smtp_server->num <
                     pDhcpcParam->smtp_server->num) ?
                     pParamList->smtp_server->num :
                     pDhcpcParam->smtp_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->smtp_server->addr [loop],
                       (char *)&pParamList->smtp_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->smtp_server->num = limit;
        }

    if (pParamList->pop3_server != NULL && pDhcpcParam->pop3_server != NULL)
        {
        limit = 0;
        if (pParamList->pop3_server->addr != NULL)
            {
            limit = (pParamList->pop3_server->num <
                     pDhcpcParam->pop3_server->num) ?
                     pParamList->pop3_server->num :
                     pDhcpcParam->pop3_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->pop3_server->addr [loop],
                       (char *)&pParamList->pop3_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->pop3_server->num = limit;
        }

    if (pParamList->nntp_server != NULL && pDhcpcParam->nntp_server != NULL)
        {
        limit = 0;
        if (pParamList->nntp_server->addr != NULL)
            {
            limit = (pParamList->nntp_server->num <
                     pDhcpcParam->nntp_server->num) ?
                     pParamList->nntp_server->num :
                     pDhcpcParam->nntp_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->nntp_server->addr [loop],
                       (char *)&pParamList->nntp_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->nntp_server->num = limit;
        }

    if (pParamList->dflt_www_server != NULL &&
        pDhcpcParam->dflt_www_server != NULL)
        {
        limit = 0;
        if (pParamList->dflt_www_server->addr != NULL)
            {
            limit = (pParamList->dflt_www_server->num <
                     pDhcpcParam->dflt_www_server->num) ?
                     pParamList->dflt_www_server->num :
                     pDhcpcParam->dflt_www_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->dflt_www_server->addr [loop],
                       (char *)&pParamList->dflt_www_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->dflt_www_server->num = limit;
        }

    if (pParamList->dflt_finger_server != NULL &&
        pDhcpcParam->dflt_finger_server != NULL)
        {
        limit = 0;
        if (pParamList->dflt_finger_server->addr != NULL)
            {
            limit = (pParamList->dflt_finger_server->num <
                     pDhcpcParam->dflt_finger_server->num) ?
                     pParamList->dflt_finger_server->num :
                     pDhcpcParam->dflt_finger_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->dflt_finger_server->addr [loop],
                       (char *)&pParamList->dflt_finger_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->dflt_finger_server->num = limit;
        }

    if (pParamList->dflt_irc_server != NULL &&
        pDhcpcParam->dflt_irc_server != NULL)
        {
        limit = 0;
        if (pParamList->dflt_irc_server->addr != NULL)
            {
            limit = (pParamList->dflt_irc_server->num <
                     pDhcpcParam->dflt_irc_server->num) ?
                     pParamList->dflt_irc_server->num :
                     pDhcpcParam->dflt_irc_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->dflt_irc_server->addr [loop],
                       (char *)&pParamList->dflt_irc_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->dflt_irc_server->num = limit;
        }

    if (pParamList->streettalk_server != NULL &&
        pDhcpcParam->streettalk_server != NULL)
        {
        limit = 0;
        if (pParamList->streettalk_server->addr != NULL)
            {
            limit = (pParamList->streettalk_server->num <
                     pDhcpcParam->streettalk_server->num) ?
                     pParamList->streettalk_server->num :
                     pDhcpcParam->streettalk_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->streettalk_server->addr [loop],
                       (char *)&pParamList->streettalk_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->streettalk_server->num = limit;
        }

    if (pParamList->stda_server != NULL &&
        pDhcpcParam->stda_server != NULL)
        {
        limit = 0;
        if (pParamList->stda_server->addr != NULL)
            {
            limit = (pParamList->stda_server->num <
                     pDhcpcParam->stda_server->num) ?
                     pParamList->stda_server->num :
                     pDhcpcParam->stda_server->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->stda_server->addr [loop],
                       (char *)&pParamList->stda_server->addr [loop],
                       sizeof (struct in_addr));
                }
            }
        pParamList->stda_server->num = limit;
        }

    /* Handle multiple IP address pairs. */

    if (pParamList->policy_filter != NULL &&
        pDhcpcParam->policy_filter != NULL)
        {
        limit = 0;
        if (pParamList->policy_filter->addr != NULL)
            {
            limit = (pParamList->policy_filter->num <
                     pDhcpcParam->policy_filter->num) ?
                     pParamList->policy_filter->num :
                     pDhcpcParam->policy_filter->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy( (char *)&pDhcpcParam->policy_filter->addr [loop * 2],
                      (char *)&pParamList->policy_filter->addr [loop * 2],
                      2 * sizeof (struct in_addr));
                }
            }
        pParamList->policy_filter->num = limit;
        }

    if (pParamList->static_route != NULL &&
        pDhcpcParam->static_route != NULL)
        {
        limit = 0;
        if (pParamList->static_route->addr != NULL)
            {
            limit = (pParamList->static_route->num <
                     pDhcpcParam->static_route->num) ?
                     pParamList->static_route->num :
                     pDhcpcParam->static_route->num;
            for (loop = 0; loop < limit; loop++)
                {
                bcopy ( (char *)&pDhcpcParam->static_route->addr [loop * 2],
                       (char *)&pParamList->static_route->addr [loop * 2],
                       2 * sizeof (struct in_addr));
                }
            }
        pParamList->static_route->num = limit;
        }

    /* Handle lists. */

    if (pParamList->vendlist != NULL && pDhcpcParam->vendlist != NULL)
        {
        limit = (pParamList->vendlist->len < pDhcpcParam->vendlist->len) ?
                pParamList->vendlist->len : pDhcpcParam->vendlist->len;
        bcopy (pDhcpcParam->vendlist->list, pParamList->vendlist->list,
               limit);
        pParamList->vendlist->len = limit;
        }

    return;
    }

/******************************************************************************
*
* dhcpcLeaseCleanup - remove data structures used by a DHCP lease
*
* This routine removes all lease-specific data structures accessed by the
* <pLeaseData> parameter. It is called internally when a lease is relinquished
* by the user with the dhcpcRelease() routine and when the DHCP client
* library is disabled with dhcpcShutdown().
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void dhcpcLeaseCleanup
    (
    LEASE_DATA * 	pLeaseData 	/* lease-specific status information */
    )
    {
    wdCancel (pLeaseData->timer);
    wdDelete (pLeaseData->timer);

    semDelete (pLeaseData->leaseSem);

    if (pLeaseData->leaseReqSpec.clid != NULL)
        {
        if (pLeaseData->leaseReqSpec.clid->id != NULL)
            free (pLeaseData->leaseReqSpec.clid->id);
        free (pLeaseData->leaseReqSpec.clid);
        }

    if (pLeaseData->leaseReqSpec.classId != NULL)
        {
        if (pLeaseData->leaseReqSpec.classId->id != NULL)
            free (pLeaseData->leaseReqSpec.classId->id);
        free (pLeaseData->leaseReqSpec.classId);
        }

    clean_param (pLeaseData->dhcpcParam);
    if (pLeaseData->dhcpcParam != NULL)
        free (pLeaseData->dhcpcParam);

    return;
    }

/******************************************************************************
*
* dhcpcCleanup - remove data structures used by DHCP client library
*
* This routine removes all data structures used by the dhcp client after
* all the lease-specific data structures have been removed. It is called
* in response to a dhcpcShutdown() request just before the client monitor
* task exits.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

LOCAL void dhcpcCleanup (void)
    {
    BOOL 		bsdDrvFlag = TRUE;
    struct ifnet * 	pIf;
    int 		level;

    /*
     * Remove all installed input hooks. Separate input hooks were installed
     * for each END-style driver, and one input hook for all BSD drivers.
     * Eventually we will be able to register DHCP as a protocol and eliminate
     * the use of input hooks. At that point, we won't need this messy access
     * to the global interface list.
     */

    level = splnet ();
    for (pIf = ifnet; pIf; pIf = pIf->if_next)
        {
        if (strcmp (pIf->if_name, "lo") == 0)  /* Skip loopback interfaces. */
            continue;
        if (muxDevExists (pIf->if_name, pIf->if_unit))
            etherInputHookDelete (dhcpcInputHook, pIf->if_name, pIf->if_unit);
        else if (bsdDrvFlag)
            {
            /* Remove the input hook for all BSD drivers. */

            etherInputHookDelete (dhcpcInputHook, NULL, pIf->if_unit);
            bsdDrvFlag = FALSE;
            }
        }
    splx (level);

    semDelete (dhcpcMutexSem);
    semDelete (dhcpcEventSem);

    rngDelete (dhcpcEventRing);
    rngDelete (dhcpcMsgRing);

    free (dhcpif.rbuf);

    free (pDhcpcRecvBuf);

    free (sbuf.buf);

    return;
    }
