/*
 * Alias_local.h contains the function prototypes for alias.c,
 * alias_db.c, alias_util.c and alias_ftp.c, alias_irc.c (as well
 * as any future add-ons).  It also includes macros, globals and
 * struct definitions shared by more than one alias*.c file.
 *
 * This include file is intended to be used only within the aliasing
 * software.  Outside world interfaces are defined in alias.h
 *
 * This software is placed into the public domain with no restrictions
 * on its distribution.
 *
 * Initial version:  August, 1996  (cjm)
 *
 * <updated several times by original author and Eivind Eklund>
 *
 * $FreeBSD: src/lib/libalias/alias_local.h,v 1.19 2001/03/25 12:05:10 brian Exp $
 */

#ifndef _ALIAS_LOCAL_H_
#define	_ALIAS_LOCAL_H_

/* Jason Wu */
#define DEBUG
typedef u_short     u_int16_t;
typedef u_long		u_int32_t;
typedef u_char      u_int8_t;



/* Macros */

/*
 * The following macro is used to update an
 * internet checksum.  "delta" is a 32-bit
 * accumulation of all the changes to the
 * checksum (adding in new 16-bit words and
 * subtracting out old words), and "cksum"
 * is the checksum value to be updated.
 */
#define	ADJUST_CHECKSUM(acc, cksum) \
	do { \
		acc += cksum; \
		if (acc < 0) { \
			acc = -acc; \
			acc = (acc >> 16) + (acc & 0xffff); \
			acc += acc >> 16; \
			cksum = (u_short) ~acc; \
		} else { \
			acc = (acc >> 16) + (acc & 0xffff); \
			acc += acc >> 16; \
			cksum = (u_short) acc; \
		} \
	} while (0)

/* Globals */

extern int packetAliasMode;

/* Prototypes */

/* General utilities */
u_short	 IpChecksum(struct ip *_pip);
u_short	 TcpChecksum(struct ip *_pip);
void	 DifferentialChecksum(u_short *_cksum, u_short *_new, u_short *_old,
	    int _n);

/* Internal data access */
struct alias_link *
	 FindIcmpIn(struct in_addr _dst_addr, struct in_addr _alias_addr,
	    u_short _id_alias, int _create);
struct alias_link *
	 FindIcmpOut(struct in_addr _src_addr, struct in_addr _dst_addr,
	    u_short _id, int _create);
struct alias_link *
	 FindFragmentIn1(struct in_addr _dst_addr, struct in_addr _alias_addr,
	    u_short _ip_id);
struct alias_link *
	 FindFragmentIn2(struct in_addr _dst_addr, struct in_addr _alias_addr,
	    u_short _ip_id);
struct alias_link *
	 AddFragmentPtrLink(struct in_addr _dst_addr, u_short _ip_id);
struct alias_link *
	 FindFragmentPtr(struct in_addr _dst_addr, u_short _ip_id);
struct alias_link *
	 FindProtoIn(struct in_addr _dst_addr, struct in_addr _alias_addr,
	    u_char _proto);
struct alias_link *
	 FindProtoOut(struct in_addr _src_addr, struct in_addr _dst_addr,
	    u_char _proto);
struct alias_link *
	 FindUdpTcpIn(struct in_addr _dst_addr, struct in_addr _alias_addr,
	    u_short _dst_port, u_short _alias_port, u_char _proto, int _create);
struct alias_link *
	 FindUdpTcpOut(struct in_addr _src_addr, struct in_addr _dst_addr,
	    u_short _src_port, u_short _dst_port, u_char _proto, int _create);
struct alias_link *
	 AddPptp(struct in_addr _src_addr, struct in_addr _dst_addr,
	    struct in_addr _alias_addr, u_int16_t _src_call_id);
struct alias_link *
	 FindPptpOutByCallId(struct in_addr _src_addr,
	    struct in_addr _dst_addr, u_int16_t _src_call_id);
struct alias_link *
	 FindPptpInByCallId(struct in_addr _dst_addr,
	    struct in_addr _alias_addr, u_int16_t _dst_call_id);
struct alias_link *
	 FindPptpOutByPeerCallId(struct in_addr _src_addr,
	    struct in_addr _dst_addr, u_int16_t _dst_call_id);
struct alias_link *
	 FindPptpInByPeerCallId(struct in_addr _dst_addr,
	    struct in_addr _alias_addr, u_int16_t _alias_call_id);
struct alias_link *
	 FindRtspOut(struct in_addr _src_addr, struct in_addr _dst_addr,
	    u_short _src_port, u_short _alias_port, u_char _proto);
struct in_addr
	 FindOriginalAddress(struct in_addr _alias_addr);
#if 1   // cnxt starts     
struct in_addr
     FindAliasAddress(struct in_addr _original_addr, struct in_addr destination_addr);
#else   // original     
struct in_addr
	 FindAliasAddress(struct in_addr _original_addr);
#endif  // #if 1     

/* External data access/modification */
int	 FindNewPortGroup(struct in_addr _dst_addr, struct in_addr _alias_addr,
                     u_short _src_port, u_short _dst_port, u_short _port_count,
		     u_char _proto, u_char _align);
void	 GetFragmentAddr(struct alias_link *_link, struct in_addr *_src_addr);
void	 SetFragmentAddr(struct alias_link *_link, struct in_addr _src_addr);
void	 GetFragmentPtr(struct alias_link *_link, char **_fptr);
void	 SetFragmentPtr(struct alias_link *_link, char *fptr);
void	 SetStateIn(struct alias_link *_link, int _state);
void	 SetStateOut(struct alias_link *_link, int _state);
int	 GetStateIn(struct alias_link *_link);
int	 GetStateOut(struct alias_link *_link);
struct in_addr
	 GetOriginalAddress(struct alias_link *_link);
struct in_addr
	 GetDestAddress(struct alias_link *_link);
struct in_addr
	 GetAliasAddress(struct alias_link *_link);
struct in_addr
	 GetDefaultAliasAddress(void);
void	 SetDefaultAliasAddress(struct in_addr _alias_addr);
u_short	 GetOriginalPort(struct alias_link *_link);
u_short	 GetAliasPort(struct alias_link *_link);
struct in_addr
	 GetProxyAddress(struct alias_link *_link);
void	 SetProxyAddress(struct alias_link *_link, struct in_addr _addr);
u_short	 GetProxyPort(struct alias_link *_link);
void	 SetProxyPort(struct alias_link *_link, u_short _port);
void	 SetAckModified(struct alias_link *_link);
int	 GetAckModified(struct alias_link *_link);
int	 GetDeltaAckIn(struct ip *_pip, struct alias_link *_link);
int	 GetDeltaSeqOut(struct ip *_pip, struct alias_link *_link);
void	 AddSeq(struct ip *_pip, struct alias_link *_link, int _delta);
void	 SetExpire(struct alias_link *_link, int _expire);
void	 ClearCheckNewLink(void);
void	 SetLastLineCrlfTermed(struct alias_link *_link, int _yes);
int	 GetLastLineCrlfTermed(struct alias_link *_link);
void	 SetDestCallId(struct alias_link *_link, u_int16_t _cid);
#ifndef NO_FW_PUNCH
void	 PunchFWHole(struct alias_link *_link);
#endif

/* Housekeeping function */
void	 HouseKeeping(void);

/* Tcp specfic routines */
/* lint -save -library Suppress flexelint warnings */

/* FTP routines */
void	 AliasHandleFtpOut(struct ip *_pip, struct alias_link *_link,
	    int _maxpacketsize);

/* IRC routines */
void	 AliasHandleIrcOut(struct ip *_pip, struct alias_link *_link,
	    int _maxsize);

/* RTSP routines */
void	 AliasHandleRtspOut(struct ip *_pip, struct alias_link *_link,
	    int _maxpacketsize);

/* PPTP routines */
void	 AliasHandlePptpOut(struct ip *_pip, struct alias_link *_link);
void	 AliasHandlePptpIn(struct ip *_pip, struct alias_link *_link);
int	 AliasHandlePptpGreOut(struct ip *_pip);
int	 AliasHandlePptpGreIn(struct ip *_pip);

/* NetBIOS routines */
int	 AliasHandleUdpNbt(struct ip *_pip, struct alias_link *_link,
	    struct in_addr *_alias_address, u_short _alias_port);
int	 AliasHandleUdpNbtNS(struct ip *_pip, struct alias_link *_link,
	    struct in_addr *_alias_address, u_short *_alias_port,
	    struct in_addr *_original_address, u_short *_original_port);

/* CUSeeMe routines */
void	 AliasHandleCUSeeMeOut(struct ip *_pip, struct alias_link *_link);
void	 AliasHandleCUSeeMeIn(struct ip *_pip, struct in_addr _original_addr);

/* Transparent proxy routines */
int	 ProxyCheck(struct ip *_pip, struct in_addr *_proxy_server_addr,
	    u_short *_proxy_server_port);
void	 ProxyModify(struct alias_link *_link, struct ip *_pip,
	    int _maxpacketsize, int _proxy_type);

enum alias_tcp_state {
	ALIAS_TCP_STATE_NOT_CONNECTED,
	ALIAS_TCP_STATE_CONNECTED,
	ALIAS_TCP_STATE_DISCONNECTED
};

void AliasHandleNetMeetingOut(struct ip *, struct alias_link *);
STATUS NatNetMeetingInit(void);
void NAT_netmeeting_queue_port(u_short);

struct alias_link *
FindNetMeetingIn(struct in_addr src_addr,   u_short src_port,
                        struct in_addr dst_addr,   u_short dst_port,
                        struct in_addr alias_addr, u_short alias_port,
                        u_char proto);

/* MS Messenger routines */
void
AliasHandleMsMsgOut(struct ip *pip, struct alias_link *link);
void
AliasHandleMsMsgIn (struct ip *pip, struct alias_link *link);



/* SIP */
void
AliasHandleSipOut(struct ip *pip, struct alias_link *link);

void
AliasHandleSipIn(struct ip *pip, struct alias_link *link);

void NAT_sip_queue_port(UINT16 portno);

STATUS NAT_sip_monitor_port(UINT16 portno);

STATUS NatSipInit(void);


/*lint -restore */

#endif /* !_ALIAS_LOCAL_H_ */
