/***************************************************************************
 *   MODULE NAME    : bridge.c                                             *
 ***************************************************************************
 *   Copyright 2001 Conexant Systems, Inc. as unpublished work             *
 *   All Rights Reserved                                                   *
 *                                                                         *
 *   The information contained herein is the confidential property         *
 *   of Conexant Systems, Inc..  The use, copying, transfer or             *
 *   disclosure of such information is prohibited except by express        *
 *   written agreement with Conexant Systems, Inc..                        *
 *                                                                         *
 *   Original Written By  :    Mickey Hung                                 *
 *   Original Date        :    02/13/2001                                  *
 ***************************************************************************/
/***************************************************************************
 * This file implements all the APIs provided for Bridging function in     *   
 * EZ-Connect project. All the data type and structure definitions are in  * 
 * "bridge.h" header file...											   * 
 ***************************************************************************/

/***************************************************************************
 * $Log: bridge.c,v $
 * Revision 1.1.1.1  2004/04/01 09:10:04  shao-gh
 * no message
 *
 * Revision 1.1.1.1  2004/01/16 09:32:58  SHAO-GH
 * no message
 *
 * Revision 1.1.1.1  2003/12/23 10:29:09  SHAO-GH
 * no message
 *
 * Revision 1.1.1.1  2003/11/21 08:29:27  SHAO-GH
 * 2003.11.20 Shao Guohua use win cvs first time from sourcesafe 
 *
 * 
 *    Rev 1.10   Jun 04 2002 11:29:18   songyh
 *  
 * 
 *    Rev 1.9   Mar 04 2002 17:32:28   fungcp
 * For DSL downstream short packet throughput 
 * performance via the DSL_DOWNSTREAM_IMPROVE 
 * macro: 
 * 
 * 1. Create the BRDG_VISA_PASS routine that
 * short cut the BR_SnarfRcvRtn call.
 * 
 * 2. Prevent MAC-forward filter out of sync with
 * MAC-table.
 * 
 * 
 *    Rev 1.8   Dec 19 2001 12:18:24   liniy
 * Added PVCS history section.
 * Moved backdoor timer check into "one_second_time_passed".
 ***************************************************************************/



/***************************************************************************/
/*								INCLUDE SECTION                            */
/***************************************************************************/
#include <Vxworks.h>
#include <taskLib.h>
#include <inetLib.h>
#include "http/httpLib.h"                
#include "http/httpRpms.h"                
#include "br_control.h"
#include "mac_table.h"
#include "bridge.h"
#include "bspcfg.h"
#include "protocol.h"
#include "br_backdoor.h"


/*------------------------------------------------------------------*/
/*    define the following options will print contents of mac_buf,  */
/*    mac_tbl to console ... for debugging purpose only             */
/*------------------------------------------------------------------*/
#undef DEBUG_MAC_BUF
#undef DEBUG_MAC_TBL






/***************************************************************************/
/*					 GLOBAL VARIABLES DECLARATION SECTION                  */
/***************************************************************************/
UINT32				BRDG_task_id;


/***************************************************************************/
/*				 	  LOCAL VARIABLES DECLARATION SECTION                  */
/***************************************************************************/
/*-----------------------------------------------*/
/*			   for circular mac buffer           */
/*-----------------------------------------------*/
LOCAL UINT8			macBuf_w_index=0;
LOCAL UINT8			macBuf_r_index=0;
STRUCT_MAC_BUF		macBuf[MAC_BUF_SIZE];


/***************************************************************************/
/*					 LOCAL FUNCTION DECLARATION SECTION                    */
/***************************************************************************/
LOCAL void		BRDG_Task(void);
LOCAL STATUS	macBuf_newPacketIn(void);
LOCAL STATUS	macBuf_getNewMac(UINT8* new_mac, BR_PORT **ppPort);

//extern unsigned long 	brBackdoorTimer;
//extern char            	brBackdoorIpStr[];
//extern char      		brBackdoorRouteIpStr[];

extern  BR_BASE  *pBrBase;

/****************************************************************************
FUNCTION NAME   : void BRDG_TaskInit(void)

DESCRIPTION     : It simply spawns the "BRDG_Task()" for continuously 
				  inspecting address table for bridging function.
				  This routine should be called at the system start-up and
				  only one time.

INPUT           : None

OUTPUT          : None

RETURN          : None
****************************************************************************/
void BRDG_TaskInit(void)
{
	/* spawn the bridging task */
	BRDG_task_id=taskSpawn(	BRDG_TASK_NAME,
							BRDG_TASK_PRIORITY,
							BRDG_TASK_PRIVATE_ENV,
							BRDG_TASK_STACK_SIZE,
							(FUNCPTR)BRDG_Task,
							0,0,0,0,0,0,0,0,0,0 );


	if( BRDG_task_id==ERROR )
		/* LogMsg here... */
	
	return;

} /* end of BRDG_TaskInit() */










/****************************************************************************
FUNCTION NAME   : LOCAL void BRDG_Task(void)

DESCRIPTION     : This routine implements the "BRDG_Task" for bridging 
				  function... This task continuously polls the "BRDG_AddrTbl", 
				  and does the followings:
				  1. Inspecting the age-time of each entry in the address table.
				     If it's timed out, then delete the entry from address table.
				  2. Do mac handling for EACH incoming packet. The code will 
				     decide to add mac in the table or simply ignore it...
				  
INPUT           : None

OUTPUT          : None

RETURN          : None
****************************************************************************/

#ifdef DSL_POLL_DEBUG	 //	AF (10/22/2001): Borrow this low level task for DSL Poll Debug
#include "dsl.h"
#define pDSL_OCDC		( volatile UINT32* )0x34002C	// HW cell loss counter
#ifdef  SYSLOG_ENABLED      
#include "logdrv.h"
#endif
#define OCDC_DISP 30 
char _buffer1[128], _buffer2[128], _buffer3[128];
int _ocdc_id=0;
int Alex_cnt1=0, Alex_cnt2=0, _amax=0, _amin=200;
int _arxisr=0, _arxh=0, _aprdy=0, _atxisr=0, _aerrisr=0, _atxu=0, _arxo=0;
//-----------------
DWORD _apNetPool[8]={0,0,0,0,0,0,0,0}, _apClPool[8]={0,0,0,0,0,0,0,0};
#endif	// DSL_POLL_DEBUG

LOCAL void BRDG_Task(void)
{
	UINT16	one_second_time_passed=( sysClkRateGet()/BRDG_TASK_POLLING_INTERVAL );
	UINT8	new_mac[6];
    BR_PORT *pPort;
	int     rtn_val;

	FOREVER
	{
		/* We use task delay time to be the time interval between two    */ 
		/* consecutive pollings... This is just an approximation,        */
		/* assuming that this task consumes no time when doing its work. */ 
		taskDelay(BRDG_TASK_POLLING_INTERVAL);

	
		/* we use this "one_second_time_passed" as an index that time has */
		/* passed one second or not. When one_second_time_passed==0 means */
		/* exactly one second time has passed, and should do something to */
		/* age timer... Each polling will make "one_second_time_passed-1" */
		/* Initially, "one_second_time_passed" will record how many       */
		/* times the FOREVER loop will do to achieve one second time      */
		/* interval...                                                    */
		one_second_time_passed--;




		/* if one second passed, adjust age timer for each entry */
		/* If time out, delete the entry...						 */
		if( one_second_time_passed==0 )
		{

            BRDG_addrTbl_Access(ADDR_TBL_ADJUST_TIMER, NULL, NULL, &rtn_val);

#if 0
			/*------ DEBUG ------*/
			d_show_mac_tbl();
			/*-------------------*/
#endif
            
            /* reset one_second_time to count from start... */
		/*****************************************************************/
		/* practical measurement found this value will make the          */
		/* "one_second" reasonable. Need to figure out the measurement.	 */	
		/*****************************************************************/ 
			one_second_time_passed=( sysClkRateGet()/BRDG_TASK_POLLING_INTERVAL );

#ifdef DSL_POLL_DEBUG
#ifdef  SYSLOG_ENABLED
if( ++_ocdc_id > OCDC_DISP ) 
{
	_ocdc_id = 0;

	SYS_Log(LOG_INFO,"*******************************\n",1,2,3,4,5,6);
	sprintf(_buffer1,"OC=%d Rx=%d Done=%d max=%d min=%d Ier=%d Iro=%d Irh=%d Iri=%d Iti=%d Itu=%d Irdy=%d\n", 
		*pDSL_OCDC, Alex_cnt1, Alex_cnt2, _amax, _amin, _aerrisr, _arxo, _arxh, _arxisr, _atxisr, _atxu, _aprdy);
	SYS_Log(LOG_INFO,_buffer1,1,2,3,4,5,6);

	#if DSL_DOWNSTREAM_IMPROVE
	if (SYS_bADSL_Support() == TRUE)
	{
		Show_Brdg_Status();
		Show_EMAC_Status();
	}
	#endif

	#ifdef MAC_OPERATION_IMPROVE
 	sprintf(_buffer1,">> NetPool: %d %d %d %d %d %d %d %d\n", _apNetPool[0], _apNetPool[1], _apNetPool[2], _apNetPool[3], _apNetPool[4], _apNetPool[5], _apNetPool[6], _apNetPool[7]);
 	SYS_Log(LOG_INFO,_buffer1,1,2,3,4,5,6);
	sprintf(_buffer1,">> ClPool: %d %d %d %d %d %d %d %d\n", _apClPool[0], _apClPool[1], _apClPool[2], _apClPool[3], _apClPool[4], _apClPool[5], _apClPool[6], _apClPool[7]);
 	SYS_Log(LOG_INFO,_buffer1,1,2,3,4,5,6);
	Show_All_Q();
	#endif
}
#endif		// SYSLOG_ENABLED
#endif		// DSL_POLL_DEBUG

            #if defined(BRDG_BACKDOOR)
            br_CheckBackdoorTimer();
            #endif
		}

		/* if new mac waited for handling, take new mac and port... */
		/* then decide to do mac learning or not...                 */
		if( macBuf_newPacketIn()==OK )
		{
			macBuf_getNewMac(new_mac, &pPort);

			BRDG_newMacHandling(new_mac, pPort);
		}

	} /* end of FOREVER */


} /* end of BRDG_Task() */










/****************************************************************************
FUNCTION NAME   : void BRDG_newMacHandling(UINT8* new_mac, BR_PORT *pPort)

DESCRIPTION     : This routine will check the new_mac against the mac address
				  table... if it's new mac then learn the mac and port, 
				  if it's already in the mac table, reset timer...
				  				  			  
INPUT           : UINT8 *new_mac	- pointer of the new mac addr. 
				  UINT8 *pPort		- pointer of the port(interface).

OUTPUT          : None

RETURN          : None
****************************************************************************/
void BRDG_newMacHandling(UINT8* new_mac, BR_PORT *pPort)
{
	UINT8	entry_index;
	int		rtn_val; 
	char 	sec[50];

	/****** notice if *new_mac and port have been changed or not ****/
	

	/* if new_mac in the table, reset timer, otherwise add new entry... */
	if( RTN_NOT_IN_TBL==BRDG_addrTbl_Access(ADDR_TBL_RESET_TIMER, 
											new_mac, 
                                            pPort,
											&rtn_val)	)
		BRDG_addrTbl_Access(ADDR_TBL_ADD_ENTRY, new_mac, pPort, &rtn_val);
	
} /* end of BRDG_newMacHandling()*/









/****************************************************************************
FUNCTION NAME   : FORWARD_DIRECTION BRDG_rxPacketHandling(  UINT8* new_mac, 
	 														BR_PORT *pPort,		
															BR_PORT** pFwdPort	)

DESCRIPTION     : This routine will determine where the received packet should 
				  be forwarded to, according the mac address of the incoming 
				  packet. The following steps will be adapted...
				  1. check if it's My Mac - pass to CPU.
				  2. check if the mac is in the mac address table - forward to
				     its corresponding port.
				  3. flood all the packet.
				  				  			  
INPUT           : UINT8 *new_mac	- pointer of the new mac addr. 
				  BR_PORT *pPort		- pointer of the port(interface) of incoming
									  packet.
				  BR_PORT** ppFwdPort - pointer of the port(interface) which indicates
				                      which port(interface) should this packet
									  be forwarded to, if the mac is in the table.
									  default return value is -1.

OUTPUT          : None

RETURN          : MY_MAC_PACKET - my mac, should forward to CPU
				  IN_TBL_PACKET - in address table, should forward to 
								  corresponding port
				  BRDCAST_PACKET - broadcast packet
				  NOT_FOUND_PACKET - none of above
****************************************************************************/
FORWARD_DIRECTION BRDG_rxPacketHandling(	UINT8 *new_mac, 
                                            BR_PORT *pSrcPort,
											BR_PORT **ppFwdPort )
{
	RTN_STATUS	status;

	/* default return value */
	*ppFwdPort = NULL;

	if( 0==memcmp(new_mac, pSrcPort->myMac, 6) )
		return MY_MAC_PACKET;
	else 
	{
		if( new_mac[0] & 0x01 )
			return BRDCAST_PACKET;

		else 
		{
			status = BRDG_addrTbl_Access(ADDR_TBL_COMPARE_MAC, 
												new_mac, 
                                                pSrcPort,
												(int *)ppFwdPort);

			/* if mac in the table, will return which port to forward this packet */
			switch ( status )
			{
				case RTN_MAC_IN_TBL:
					return IN_TBL_PACKET;
				case RTN_NOT_IN_TBL:
				default:
					return NOT_FOUND_PACKET;
			}
	
		} /* end of inner if */
	} /* end of outer if */

} /* end of BRDG_rxPacketHandling() */









/****************************************************************************
FUNCTION NAME   : LOCAL STATUS macBuf_newPacketIn(void)

DESCRIPTION     : This routine will check if there's any new mac in the 
				  (proprietary) circular queue waiting for handling or not.

INPUT           : None

OUTPUT          : None

RETURN          : OK	- there're new macs in the queue waiting for handling.
				  ERROR	- no any new mac wating for handling.
****************************************************************************/
LOCAL STATUS macBuf_newPacketIn(void)
{
	/* if macBuf_w_index!=macBuf_r_index means there're new mac been put in */
	/* the circular buffer.....	    									    */			
	if( macBuf_w_index==macBuf_r_index )
		return ERROR;				/* no new mac */
	else
		return OK;					/* new mac */

} /* end of macBuf_newPacketIn() */










/****************************************************************************
FUNCTION NAME   : LOCAL STATUS macBuf_getNewMac(char* new_mac, int* port)

DESCRIPTION     : This routine will try to get the mac address and 
				  coressponding port(interface) of which macBuf_r_index 
				  locates in the mac circular buffer. And put mac address ptr
				  in *new_mac, correpsonding port(interface) ptr in *int.   
				  If it fails, the code will put NULL into these two pointers.
				  
INPUT           : char *new_mac	- pointer of the new mac addr.
				  int *port		- pointer of the port(interface).

OUTPUT          : None

RETURN          : OK - new mac available to process
				  ERROR - no new mac available
****************************************************************************/
LOCAL STATUS macBuf_getNewMac(UINT8* new_mac, BR_PORT **ppPort)
{

	if( macBuf_r_index!=macBuf_w_index )
	{
		/* get the data pointer from macBuf */
		memmove(new_mac, macBuf[macBuf_r_index].mac, 6);
		memmove(ppPort, &(macBuf[macBuf_r_index].pPort), sizeof(VOID*) );

		macBuf_r_index++;

		/* circular buffer, wrap around */
		if( macBuf_r_index>MAC_BUF_BOTTOM )
			macBuf_r_index=MAC_BUF_TOP;

		return OK;
	}
	else
	{
		/* here means no new mac to handle */
		new_mac=NULL;
        ppPort=NULL;

		return ERROR;
	}

} /* end of macBuf_getNewMac() */











/****************************************************************************
FUNCTION NAME   : STATUS macBuf_putNewMac(UINT8* new_mac, BR_PORT *pPort)

DESCRIPTION     : This routine will try to put the mac address and 
				  coressponding port(interface) of the incoming packet into 
				  the mac circular buffer. 
				  			  
INPUT           : UINT8 	*new_mac	- pointer of the new mac addr. 
				  BR_PORT	*pPort		- pointer of the port(interface).

OUTPUT          : None

RETURN          : OK	- add mac & port successfully
				  ERROR - fails
****************************************************************************/
STATUS macBuf_putNewMac(UINT8* new_mac, BR_PORT *pPort)
{
	UINT8 temp_index=macBuf_w_index;

	temp_index++;

	if( temp_index>MAC_BUF_BOTTOM )
		temp_index=0;


	/* if after adjusted, 2 indexes are not the same, */
	/* we can still add new mac...				      */
	/* to prevent w_index has one cycle faster than   */
	/* r_index...									  */
	if( temp_index!=macBuf_r_index )
	{
		memmove( macBuf[macBuf_w_index].mac, new_mac, 6 );
		memmove( &(macBuf[macBuf_w_index].pPort), &pPort, sizeof(VOID *) );

		macBuf_w_index=temp_index;

		return OK;
	}

	return ERROR;

} /* end of macBuf_putNewMac() */








/****************************************************************************
FUNCTION NAME   : void d_show_mac_buf(void)

DESCRIPTION     : This routine will print global variables(to console):
                    macBuf_r_index,
                    macBuf_w_index,
                  and contents of macBuf[]...
				  For debugging purpose only...			  

INPUT           : None

OUTPUT          : None

RETURN          : None
****************************************************************************/
void d_show_mac_buf(void)
{
	int 	i=0;
	int 	j=0;


	/*----- DEBUG -----*/
	printf("macBuf_r_index=%2d\n",macBuf_r_index);	
	printf("macBuf_w_index=%2d\n\n",macBuf_w_index);
	
	printf("******* %s *******\n", "Contents of macBuf[MAC_BUF_SIZE]");
	
	for( i=0; i<MAC_BUF_SIZE; i++)
	{
		printf("MAC ADDR: %2x %2x %2x %2x %2x %2x\n",	
												macBuf[i].mac[0], 
												macBuf[i].mac[1],
												macBuf[i].mac[2], 
												macBuf[i].mac[3], 
												macBuf[i].mac[4], 
												macBuf[i].mac[5]    );

		printf("PORT POINTER=%4d\n\n", (UINT32)macBuf[i].pPort );
	} /* end of for */

	
} /* end of d_show_mac_buf */







/***************************************************************************/
/* $History$
 */



