/***************************************************************************** 
FILE NAME:	Protocol.h:

HISTORY:	98.11.16	Creation		G.L.
		98.11.19	Snarf Provisioning	G.L.
		98.11.29	Transparent Bridging	G.L.
		98.12.04	Address Learning	G.L.
		99.03.09	Release 0.1		G.L.

DESCRIPTION:	This header file defines constants, data types and function
		prototypes for the Snarf protocols of the SuperPIPE product.
		The implemention of functions is contained in Protocol.c. 

REFERENCES:	"SuperPIPE ARM#1 Software Design Document"
		"SuperPIPE Silicon System Design Engineering Specification" 

		Copyright 1999 Conexant Systems, Inc. 
		ALL RIGHTS RESERVED
		This file is Conexant proprietary, and all Conexant corporate 
		policies regarding source-file distribution must be observed.
********************************************************************************/

#ifndef __PROTOCOL_H
#define __PROTOCOL_H

#ifdef __cplusplus
extern "C" {
#endif


#include "vxWorks.h"
#include "net/mbuf.h"
#include "netBufLib.h"  
#include "net/systm.h"
#include "netLib.h"
#include "sys/ioctl.h"
#include "sysLib.h"
#include "sys/times.h"
#include "errno.h"
#include "errnoLib.h"
#include "memLib.h"
#include "intLib.h"
#include "cacheLib.h"
#include "logLib.h"
#include "iosLib.h"
#include "stdio.h"
#include "stdlib.h"
#include "string.h"
#include "muxLib.h"
#include "etherLib.h"
#include "end.h"
//#include "emac_end.h"




/********************************
 ****** CONSTANTS and MACOS *****
 ********************************/
#define NAT_E	1
#define NAPT_E	2
#define M_IFFWD 0x08
#define	MAX_INTERFACE_NAME_LEN	32
#define MAC_ADDRESS_LEN		    6             
#define MAX_SEND_FAILS			100	/* Max number of fails before mark the dev down */
#define MAX_802_2_LENGTH        1500

/* IEEE 802.2 XID and TEST command */
#define XID_PDU            0xAF /* XID command defined as  1 1 1 1  P 1 0 1  with LSBit first */
#define TEST_PDU           0xE3 /* TEST command defined as 1 1 0 0  P 1 1 1  with LSBit first */
#define P_F_BIT            0x10 /* Poll or Final bit */
#define SSAP_C_R_BIT       0x01 /* Command/Response bit in SSAP field */
#define DSAP_I_G_BIT       0x01 /* Individual/Group bit in DSAP field */
#define XID_FORMAT_ID      0x81 /* XID format identifier */
#define LLC_TYPE_1         0x01 /* Type 1 LLC */
#define RECEIVE_WND_SIZE_R 0x00 /* XID sender's receive window size */ 

/* Error codes for Snarf protocols */
#define CMAC_SNARF_ERR_0			0
#define CMAC_SNARF_MUXSEND_ERR		SNARF_ERROR_0 + 1		
#define CMAC_SNARF_ADDR_ADD_ERR		CMAC_SNARF_MUXSEND_ERR + 1
#define CMAC_SNARF_NET_POOL_INIT_ERR	CMAC_SNARF_ADDR_ADD_ERR + 1
#define CMAC_SNARF_MUXBIND_ERR		CMAC_SNARF_NET_POOL_INIT_ERR + 1
#define CMAC_SNARF_DEV_NAME_LEN_ERR	CMAC_SNARF_MUXBIND_ERR + 1
#define CMAC_SNARF_CL_POOL_GET_ERR	CMAC_SNARF_DEV_NAME_LEN_ERR +1

#define EMAC_SNARF_ERROR_0		CMAC_SNARF_DEV_NAME_LEN_ERR +1
#define EMAC_SNARF_MUXSEND_ERR		EMAC_SNARF_ERR_0 + 1
#define EMAC_SNARF_ADDR_TAB_MAX_ERR	EMAC_SNARF_MUXSEND_ERR + 1
#define EMAC_SNARF_ADDR_ADD_ERR		EMAC_SNARF_MUXSEND_ERR + 1
#define EMAC_SNARF_ADDR_TAB_DEL_ERR	EMAC_SNARF_ADDR_ADD_ERR + 1
#define EMAC_SNARF_NET_POOL_INIT_ERR	EMAC_SNARF_ADDR_TAB_DEL_ERR + 1
#define EMAC_SNARF_MUXBIND_ERR		EMAC_SNARF_NET_POOL_INIT_ERR + 1
#define EMAC_SNARF_DEV_NAME_LEN_ERR	EMAC_SNARF_MUXBIND_ERR + 1
#define EMAC_SNARF_CL_POOL_GET_ERR	EMAC_SNARF_DEV_NAME_LEN_ERR +1
	
#define SNARF_DEBUG
#define sfDbMsg(s) printf("%s \n", s)		/* Print Debug messages */ 

#if 0
#define EMAC_FIX_MUX_ALIGNMENT_BUG_IN_SNARF
#endif

#if 1
#define CMAC_FIX_MUX_ALIGNMENT_BUG_IN_SNARF
#endif

#define MUX_BUG_PATCH_CL_OFFSET		ENET_HDR_REAL_SIZ 	/* 2 - Byte offset to fix a MUX bug */

/********************************
 *** DATA TYPES AND VARIABLES ***
 ********************************/

/* This structure stores information of the END driver to which a protocol is
   bound. Each END driver is identified in two ways: a) by the interface name
   and the unit number of the interface; b) by the cookie of the END driver.
   The cookie is a MUX internal structure , which is returned when the protocol
   is bound to the END and it is used by MUX to identify the END.

   Note: For the initial release of SuperPIPE, this structure is used by
   the CMAC Snarf protocol and the EMAC Snarf protocol to store information
   about the CMAC END and EMAC END, respectively. But, in future, it can also
   be used for other protocols and END drivers (e.g. USB and teleco return).*/
typedef struct
   { char		   name[MAX_INTERFACE_NAME_LEN];	/* Interface name */ 
	  int		      unit;		      /* Interface unit number */
	  void         *pCookie;	   /* Cookie for END driver  */
	  END_OBJ      *pDrvHandle;	/* Handle for END driver  */
	  NET_POOL_ID	pNetPool;	   /* Net pool id allocated by END */
	  CL_POOL_ID	pClPool;	      /* Cluster pool id allocated by END */
	  int			   nSendFails;    /* # endSend failures in Snarf */
	  BOOL		   devOK;		   /* Interface device is operational */
	  NET_POOL_ID	pNetSave;	   /* Save Net pool id for bridging _tamd */
	} END_DRIVER_INFO;	

/* Generic MAC address */
#ifndef _mac_address_type
#define _mac_address_type
typedef UINT8	MAC_ADDRESS_TYPE[MAC_ADDRESS_LEN];
#endif

/* Generic Ethernet header format */
typedef struct 
	{
	  MAC_ADDRESS_TYPE	da;		/* Destination address */
	  MAC_ADDRESS_TYPE	sa;		/* Source address      */
	  UINT16		type_len;	/* Type or length      */
	}   ETHERNET_HDR;

/* SNMP support */
typedef struct {
    /* IEEE 802.3 MAC fields */
    UINT8  ucDest[MAC_ADDRESS_LEN];   /* 0-5 */
    UINT8  ucSource[MAC_ADDRESS_LEN]; /* 6-11 */
    UINT16 usLength;                  /* 12-13 */
    /* IEEE 802.2 LLC fields */
    UINT8  ucDsap;                    /* 14 */
    UINT8  ucSsap;                    /* 15 */
    UINT8  ucControl;                 /* 16 */
    /* IEEE 802.2 SNAP fields */
    UINT8  ucXIDFormatID;             /* 17 */
    UINT8  ucLLCTypeInfo;             /* 18 */
    UINT8  ucReceiveWindowSize;       /* 19 */        
    UINT8  ucType[2];                 /* 20-21 */
    /* data */
} LLC_PDU;

typedef struct end_driver_control
{
    END_OBJ     		end;			/* The class we inherit from. */
    unsigned int    	bridgePort;
} END_DRIVER_CONTROL;


extern UINT8 NULL_MAC_ADDR	[MAC_ADDRESS_LEN];
extern UINT8 BROADCAST_MAC_ADDR[MAC_ADDRESS_LEN];

extern MAC_ADDRESS_TYPE	CMAC_ADDR;  /* CMAC Addr */
extern MAC_ADDRESS_TYPE	EMAC_ADDR;  /* EMAC Addr */

//extern int SF_EIOCAUCASTADDR; /* NOTE: To be removed once the global h file containing */
//extern int SF_EIOCDUCASTADDR; /* these constants are available. */

/*******************************
 ***** FUNCTION PROTOTYPES *****
 *******************************/

/* These functions are provided for OEM to initialize the EMAC/CMAC MAC addresses
and the mac Address Table with provisioned CPE addresses. */
int snarfInitCmacAddr(MAC_ADDRESS_TYPE  addr);
int snarfInitEmacAddr(MAC_ADDRESS_TYPE  addr);
int snarfInitMacAddresTable (UINT16 maxTableSize);  	/* Allocate snarfMacAddrTable.*/
int snarfAddMacAddress (MAC_ADDRESS_TYPE  addr);	/* Add address to the address table */
int snarfDelMacAddress (MAC_ADDRESS_TYPE  addr);	/* Delete address from the table */



/************************************************************* 
Standard protocol routines required by MUX.                                
There are two snarf protocols for this project. One is bound  
to CMAC END; the other is bound to EMAC END.                  
**************************************************************/


/* ---------------------------------
   Snarf Protocol Bound to CMAC END 

   Protocol Name: cmacSnarfProtocol
   Protocol Type: MUX_PROTO_SNARF   
   ---------------------------------*/

/* Receive routine for MUX to pass packet to cmacSnarfProtocol
   The main processing of the protocol takes place here.   
   The processing includes:
      - transparent bridging
      - LLC/IP Filtering                                        */
BOOL cmacSnarfRcvRtn
	(void* 		pCookie,	/* returned by muxBind() call */ 
         long 		type, 		/* protocol type (MUX_PROTO_SNARF) */
         M_BLK_ID 	pNetBuff, 	/* packet with link info */
         LL_HDR_INFO 	*pLinkHdr, 	/* link-level header info */
         void* 		pSpare);	/* spare info structure (Not Used) */

/* MUX uses this routine to gracefully shut down cmacSnarfProtocol. */   
int cmacSnarfShutDownRtn 
	(void*		pCookie,         /* returned by muxBind() call */
	 void* 		pSpare);         /* spare info structure (Not Used) */

/* MUX uses this routine to restart the protocol that had to stop
   transmitting because the device was out of resource. This routine
   will make the protocol resume the transmission to CMAC.*/
int cmacSnarfTxRestartRtn  
	(void*		pCookie,        /* returned by muxBind() call */
	 void*		pSpare);	

/* MUX uses this routine to pass error messages from CMAC to protocol.
   This routine has appropriate responses for all possible error messages */
void cmacSnarfErrorRtn
	(END_OBJ* 	pEnd,		/* returned by muxBind() call */
 	 END_ERR* 	pError,		/* pointer to error structure */
	 void*		pSpare);	/* spare info structure (Not Used) */


/* ---------------------------------
   Snarf Protocol Bound to EMAC END 

   Protocol Name: emacSnarfProtocol
   Protocol Type: MUX_PROTO_SNARF   
   ---------------------------------*/

/* Receive routine for MUX to pass packet to emacSnarfProtocol
   The main processing of the protocol takes place here.   
   The processing includes:
      - transparent bridging
      - CPE MAC address learning
      - LLC/IP Filtering                                        */
BOOL emacSnarfRcvRtn
	(void* 		pCookie,	/* returned by muxBind() call */ 
         long 		type, 		/* protocol type (MUX_PROTO_SNARF) */
         M_BLK_ID 	pNetBuff, 	/* packet with link info */
         LL_HDR_INFO 	*pLinkHdr, 	/* link-level header info */
         void* 		pSpare);	/* spare info structure (Not Used) */

/* MUX uses this routine to gracefully shut down emacSnarfProtocol. */   
int emacSnarfShutDownRtn 
	(void*		pCookie,         /* returned by muxBind() call */
	 void* 		pSpare);         /* spare info structure (Not Used) */

/* MUX uses this routine to restart the protocol that had to stop
   transmitting because the device was out of resource. This routine
   will make the protocol resume the transmission to EMAC.*/
int emacSnarfTxRestartRtn  
	(void*		pCookie,	/* returned by muxBind() call */
	 void*		pSpare);	

/* MUX uses this routine to pass error messages from EMAC to protocol.
   This routine has appropriate responses for all possible error messages */
void emacSnarfErrorRtn
	(END_OBJ* 	pEnd,		/* returned by muxBind() call */
 	 END_ERR* 	pError,		/* pointer to error structure */
	 void*		pSpare);	/* spare info structure (Not Used) */


/******************************************************************* 
Initialization routines for cmacSnarfProtocol and emacSnarfProtocol.   
They are called by tUsrRoot. They bind snarf protocols to END drivers,
and save any necessary info related to the drivers.
Note that tUsrRoot has to provide the name and unit number for END
drivers. 
********************************************************************/

int initCmacSnarfProtocol
	(char	*pName,		/* Interface name for CMAC */
	 int	unit);		/* Interface unit number for CMAC */
 
int initEmacSnarfProtocol
	(char	*pName,		/* Interface name for EMAC */
	 int	unit);		/* Interface unit number for EMAC */



/**************
LLC/IP Filters 
**************/

/* LLC/IP Filter for CMAC */
BOOL cmacSnarfLlcIpFilter (M_BLK_ID pNetBuff, LL_HDR_INFO  *pLinkHdr);

/* LLC/IP Filter for EMAC */
BOOL emacSnarfLlcIpFilter (M_BLK_ID pNetBuff, LL_HDR_INFO  *pLinkHdr);


/**********************
Fix a MUX Alignment Bug
**********************/
int snarfClusterAlignment 
       (  
	END_DRIVER_INFO *pMacInfo,   /* Driver info */
	M_BLK_ID	pNetBuff,    /* Original received packet */
	M_BLK_ID	pTempMblk,   /* Newly allocated and aligned packet */
	CL_BLK_ID	pTempClBlk,  /* Newly allocated CL Blk */
	char*		pTempCl      /* Newly allocated CL */
       );


#ifdef __cplusplus
}
#endif

#endif   /* __PROTOCOL_H */
