/* dhcpc_subr.c - DHCP client general subroutines */

/* Copyright 1984 - 1999 Wind River Systems, Inc. */
#include "copyright_wrs.h"

/*
modification history
____________________
01o,10sep99,gq   ignore unsupported DHCP message options
01n,01mar99,spm  eliminated creation of invalid route (SPR #24266)
01m,04dec97,spm  added code review modifications
01l,04dec97,spm  replaced muxDevExists call with test of predefined flag;
                 enabled ARP test of intended IP address
01k,06oct97,spm  removed reference to deleted endDriver global; replaced with
                 support for dynamic driver type detection
01j,26aug97,spm  major overhaul: reorganized code and changed user interface
                 to support multiple leases at runtime
01i,06aug97,spm  renamed class field of dhcp_reqspec structure to prevent C++
                 compilation errors (SPR #9079)
01h,15jul97,spm  set sa_len and reordered ioctl calls in configuration routine,
                 changed to raw sockets to avoid memPartFree error (SPR #8650);
                 replaced floating point to prevent ss5 exception (SPR #8738)
01g,10jun97,spm  corrected load exception (SPR #8724) and memalign() parameter;
                 isolated incoming messages in state machine from input hooks
01f,02jun97,spm  changed DHCP option tags to prevent name conflicts (SPR #8667)
                 and updated man pages
01e,09may97,spm  changed memory access to align IP header on four byte boundary
01d,18apr97,spm  added conditional include DHCPC_DEBUG for displayed output,
                 shut down interface when changing address/netmask
01c,07apr97,spm  corrected bugs extracting options, added safety checks and
                 memory cleanup code, rewrote documentation
01b,27jan97,spm  added little-endian support and modified to coding standards
01a,03oct96,spm  created by modifying WIDE Project DHCP Implementation
*/

/*
DESCRIPTION
This library contains subroutines which implement several utilities for the
Wide project DHCP client, modified for vxWorks compatibility.

INTERNAL
The routines in this module are called by both portions of the runtime state
machine, in the dhcpcState1 and dhcpcState2 modules, which execute before and
after a lease is established, as well as the boot-time (partial) state machine
in the dhcpcBoot module.

INCLUDE_FILES: dhcpcLib.h
*/

/*
 * WIDE Project DHCP Implementation
 * Copyright (c) 1995 Akihiro Tominaga
 * Copyright (c) 1995 WIDE Project
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided only with the following
 * conditions are satisfied:
 *
 * 1. Both the copyright notice and this permission notice appear in
 *    all copies of the software, derivative works or modified versions,
 *    and any portions thereof, and that both notices appear in
 *    supporting documentation.
 * 2. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by WIDE Project and
 *      its contributors.
 * 3. Neither the name of WIDE Project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPER ``AS IS'' AND WIDE
 * PROJECT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
 * WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE. ALSO, THERE
 * IS NO WARRANTY IMPLIED OR OTHERWISE, NOR IS SUPPORT PROVIDED.
 *
 * Feedback of the results generated from any improvements or
 * extensions made to this software would be much appreciated.
 * Any such feedback should be sent to:
 *
 *  Akihiro Tominaga
 *  WIDE Project
 *  Keio University, Endo 5322, Kanagawa, Japan
 *  (E-mail: dhcp-dist@wide.ad.jp)
 *
 * WIDE project has the rights to redistribute these changes.
 */

/* includes */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

#include <sys/types.h>
#include <time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>
#include <inetLib.h>

#include <etherLib.h>
#include <sysLib.h>
#include <logLib.h>
#include <sockLib.h>
#include <ioLib.h>
#include <semLib.h>
#include <vxLib.h>
#include "muxLib.h"

#include "dhcp/dhcp.h"
#include "dhcp/dhcpcStateLib.h"
#include "dhcp/dhcpcCommonLib.h"
#include "dhcp/dhcpcInternal.h"

#define CHKOFF(LEN)  (offopt + 2 + (LEN) < DFLTOPTLEN) /* test option offset */

/* globals */

IMPORT SEM_ID dhcpcArpEventSem;
IMPORT u_char dhcpArpSpa[4];
IMPORT BOOL dhcpArpInputHook (struct ifnet*, char*, int);
IMPORT int flushroutes (void);
IMPORT char * 	pDhcpcRecvBuf; 		/* Message buffer for state machine. */

struct msg dhcpcMsgOut;
struct msg dhcpcMsgIn;
struct ps_udph spudph;

/* forward declarations */

int config_if();
void set_route();
void make_decline (char *, struct dhcp_reqspec *, struct if_info *);
void make_release (struct dhcp *, struct dhcp_reqspec *, struct if_info *);
long generate_xid (struct if_info *);

/*******************************************************************************
*
* generate_xid - generate a transaction identifier
*
* This routine forms a transaction identifier which is used by the lease
* for transmissions over the interface described by <pIfData>. The routine
* is called from multiple locations after the initialization routines have
* retrieved the network interface hardware address.
*
* RETURNS: 32-bit transaction identifier in host byte order.
*
* ERRNO: N/A
*
* NOMANUAL
*/

long generate_xid
    (
    struct if_info * 	pIfData 	/* interface used by lease */
    )
    {
    time_t current = 0;
    u_short result1 = 0;
    u_short result2 = 0;
    u_short result3 = 0;

    u_short tmp[6];

    bcopy (pIfData->haddr.haddr, (char *)tmp, 6);

    time (&current);
    result1 = checksum (tmp, 6);
    result2 = checksum ( (u_short *)&current, 2);

    /*
     * Although unlikely, it is possible that separate leases which are using
     * the same network interface calculate a checksum at the same time. To
     * guarantee uniqueness, include the value of the lease-specific pointer
     * to the interface descriptor.
     */

    tmp [0] = ((long)pIfData) >> 16;
    tmp [1] = ((long)pIfData) & 0xffff;

    result3 = checksum (tmp, 4);

    return ( (result1 << 16) + result2 + result3);
    }

/*******************************************************************************
*
* arp_check - use ARP to check if given address is in use
*
* This routine broadcasts an ARP request and waits for a reply to determine if
* an IP address offered by a DHCP server is already in use.
*
* RETURNS: ERROR if ARP indicates client in use, or OK otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int arp_check
    (
    struct in_addr *target,
    struct if_info *pIfData 	/* interface used by lease */
    )
    {
    int i = 0;
    long tmp;
    char inbuf [ETHERHL + sizeof (struct ether_arp)];
    struct ether_header *sether = NULL;
    struct ether_arp *sarp = NULL;
    STATUS result;

    bzero (inbuf, ETHERHL + sizeof (struct ether_arp));
    sether = (struct ether_header *) inbuf;
    sarp = (struct ether_arp *) &inbuf [ETHERHL];

    sarp->arp_hrd = htons (ARPHRD_ETHER);
    sarp->arp_pro = htons (ETHERTYPE_IP);
    sarp->arp_hln = 6;
    sarp->arp_pln = 4;
    sarp->arp_op = htons (ARPOP_REQUEST);

    /* Set sender H/W address to your address for ARP requests. (RFC 1541). */
    for (i = 0; i < 6; i++)
        {
        sarp->arp_sha[i] = sether->ether_shost[i] =
                           pIfData->haddr.haddr[i];
        sarp->arp_tha[i] = 0;
        sether->ether_dhost[i] = 0xff;
        }

    /* Set sender IP address to 0 for ARP requests as per RFC 1541. */
    tmp = 0;
    bcopy ( (char *)&tmp, (char *)sarp->arp_spa, sarp->arp_pln);
    bcopy ( (char *)&target->s_addr, (char *)sarp->arp_tpa, sarp->arp_pln);
    sether->ether_type = ETHERTYPE_ARP;

    if (etherOutput (pIfData->iface, sether, (char *)sarp,
                     sizeof (struct ether_arp)) == ERROR)
        {
        return (OK);                      /* it is harmless to return 0 */
        }

    /* Set sender address to our address for checking by etherHook. */
    bcopy ( (char *)&target->s_addr, (char *)dhcpArpSpa, sarp->arp_pln);

    /* Create signalling semaphore. */
    dhcpcArpEventSem = semBCreate (SEM_Q_FIFO, SEM_EMPTY);
    if (dhcpcArpEventSem == NULL)
        {
#ifdef DHCPC_DEBUG
        logMsg ("Error creating ARP semaphore.\n", 0, 0, 0, 0, 0, 0);
#endif
        return (OK);
        }

    /* Wait up to one-half a second for a reply. If none received, return 0. */

    if (pIfData->endDrvFlag)
        etherInputHookAdd (dhcpArpInputHook, pIfData->name, pIfData->unit);
    else
        etherInputHookAdd (dhcpArpInputHook, NULL, pIfData->unit);

    result = semTake (dhcpcArpEventSem, sysClkRateGet() / 2);

    semDelete (dhcpcArpEventSem);

    if (pIfData->endDrvFlag)
        etherInputHookDelete (dhcpArpInputHook, pIfData->name, pIfData->unit);
    else
        etherInputHookDelete (dhcpArpInputHook, NULL, pIfData->unit);

    if (result == ERROR)    /* Error indicates timeout occurred. */
        return (OK);
    else                    /* ARP reply received - duplicate IP address. */
        return (ERROR);
    }

/*******************************************************************************
*
* arp_reply - use ARP to notify other hosts of new address in use
*
* This routine broadcasts an ARP reply when the DHCP client changes addresses
* to a new value retrieved from a DHCP server.
*
* RETURNS: -1 if ARP indicates client in use, or 0 otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int arp_reply
    (
    struct in_addr *ipaddr,
    struct if_info *pIfData 	/* interface used by lease */
    )
    {
    int i = 0;
    char inbuf [ETHERHL + sizeof (struct ether_arp) ];
    struct ether_header *sether = NULL;
    struct ether_arp *sarp = NULL;

    bzero (inbuf, ETHERHL + sizeof (struct ether_arp));
    sether = (struct ether_header *)inbuf;
    sarp = (struct ether_arp *)&inbuf [ETHERHL];

    sarp->arp_hrd = htons (ARPHRD_ETHER);
    sarp->arp_pro = htons (ETHERTYPE_IP);
    sarp->arp_hln = 6;
    sarp->arp_pln = 4;
    sarp->arp_op = htons (ARPOP_REPLY);

    for (i = 0; i < 6; i++)
        {
        sarp->arp_tha[i] = sarp->arp_sha[i] = sether->ether_shost[i]
                         = pIfData->haddr.haddr[i];
        sether->ether_dhost[i] = 0xff;
        }

    bcopy ( (char *)&ipaddr->s_addr, (char *)sarp->arp_spa, sarp->arp_pln);
    bcopy ( (char *)&ipaddr->s_addr, (char *)sarp->arp_tpa, sarp->arp_pln);

    sether->ether_type = ETHERTYPE_ARP;

    if (etherOutput (pIfData->iface, sether, (char *)sarp,
                     sizeof (struct ether_arp)) == ERROR)
        return (-1);

    return (0);
    }

/*******************************************************************************
*
* nvttostr - convert NVT ASCII to strings
*
* This routine implements a limited NVT conversion which removes any embedded
* NULL characters from the input string.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

int nvttostr
    (
    char *	nvt,
    char *	str,
    int 	length
    )
    {
    FAST int i = 0;
    FAST char *tmp = NULL;

    tmp = str;

    for (i = 0; i < length; i++)
        {
        if (nvt[i] != NULL)
            {
            *tmp = nvt[i];
            tmp++;
            }
        }

    str [length] = '\0';
    return (0);
    }

/*******************************************************************************
*
* align_msg - set the buffer pointers to access message components
*
* This routine sets the pointers in the given message descriptor
* structure to access the various components of a received DHCP
* message. It is used internally by the Ethernet input hook and
* the state machine.
*
* RETURNS: N/A
*
* ERRNO:   N/A
*
* NOMANUAL
*/

void align_msg
    (
    struct msg * 	rmsg, 	/* Components of received message */
    char * 		rbuf 	/* Received Ethernet packet */
    )
    {
    rmsg->ether = (struct ether_header *) rbuf;
    rmsg->ip = (struct ip *) &rbuf[ETHERHL];

    if ( (ntohs (rmsg->ip->ip_off) & 0x1fff) == 0 &&
        ntohs (rmsg->ip->ip_len) >= DFLTBOOTPLEN + UDPHL + IPHL)
        {
#if BSD<44
        rmsg->udp = (struct udphdr *)&rbuf [ETHERHL +
                                         (rmsg->ip->ip_v_hl & 0xf) * WORD_LEN];
        rmsg->dhcp = (struct dhcp *)&rbuf [ETHERHL +
                                        (rmsg->ip->ip_v_hl & 0xf) * WORD_LEN +
                                         UDPHL];
#else
        rmsg->udp = (struct udphdr *) &rbuf [ETHERHL +
                                           rmsg->ip->ip_hl * WORD_LEN];
        rmsg->dhcp = (struct dhcp *) &rbuf [ETHERHL +
                                           rmsg->ip->ip_hl * WORD_LEN +
                                          UDPHL];
#endif
        }
    else
        {
        rmsg->udp = NULL;
        rmsg->dhcp = NULL;
        }

    return;
    }

/*******************************************************************************
*
* dhcp_msgtoparam - expand DHCP message into data structure
*
* This routine converts a DHCP message from the transmission format into the
* dhcp_param structure. It copies the core fields directly and multiplexes
* the options to the appropriate handle_* functions.
*
* RETURNS: 0 if conversion successful, or -1 otherwise
*
* ERRNO: N/A
*
* INTERNAL
*
* When calculating the values for the lease timers, floating-point calculations
* can't be used because some boards (notably the SPARC architectures) disable
* software floating point by default to speed up context switching. These
* boards abort with an exception when floating point operations are
* encountered. The error introduced by the integer approximations is not
* significant.
*
* NOMANUAL
*/

int dhcp_msgtoparam
    (
    struct dhcp *msg,
    int msglen,
    struct dhcp_param *parameter
    )
    {
    FAST unsigned char *optp = NULL;
    unsigned char tag = 0;
    BOOL sname_is_opt = FALSE;
    BOOL file_is_opt = FALSE;
    int err = 0;
    unsigned char *endofopt;

    endofopt = &msg->options [msglen - DFLTDHCPLEN + DFLTOPTLEN];

    bzero (parameter->got_option, GOTOPTSIZ);

    for (optp = &msg->options [MAGIC_LEN]; optp <= endofopt; optp++)
        {
        tag = *optp;

#ifdef DHCPC_DEBUG
    logMsg ("dhcp_msgtoparam:processing tag(%d[%d])\n", (unsigned int)tag, (unsigned int)*(optp+1), 0, 0, 0, 0);
#endif

        /* skip the PAD option */

        if (tag == _DHCP_PAD_TAG)
            continue;

        /* stop processing when the END option is encountered */

        if (tag == _DHCP_END_TAG)
            break;

        /* ignore unsupported DHCP message options */

        if (tag > _DHCP_LAST_OPTION)
            {
#ifdef DHCPC_DEBUG
            logMsg ("dhcp_msgtoparam:tag(%d) >= _DHCP_LAST_OPTION(%d)\n", tag, _DHCP_LAST_OPTION, 0, 0, 0, 0);
#endif
            optp++;
            optp += *optp;
            continue;
            }

        /* handle the "Option Overload" */
        if (tag == _DHCP_OPT_OVERLOAD_TAG)
            {
            optp += 2;
            switch (*optp)
                {
                case FILE_ISOPT:
                    file_is_opt = TRUE;
	            break;
                case SNAME_ISOPT:
                    sname_is_opt = TRUE;
                    break;
                case BOTH_AREOPT:
                    file_is_opt = sname_is_opt = TRUE;
                    break;
                default:
                    break;
                }
            continue;
            }

        if (handle_param [ (unsigned int)tag] != NULL)
            {
            if ( (err = (*handle_param [ (unsigned int)tag]) (optp, parameter)) != 0)
                return (err);
            else
                SETBIT (parameter->got_option, tag);
            }

        /* Set the message type tag to distinguish DHCP and BOOTP messages. */
        else if (tag == _DHCP_MSGTYPE_TAG)
            SETBIT (parameter->got_option, tag);

        optp++;
        optp += *optp;
        }

    if (file_is_opt)
        {
        endofopt = &msg->file [MAX_FILE];
        for (optp = msg->file; optp <= endofopt; optp++)
            {
            tag = *optp;

            /* skip the PAD option */
            if (tag == _DHCP_PAD_TAG)
                continue;

            /* stop processing when the END option is reached */

            if (tag == _DHCP_END_TAG)
                break;

            if (tag > _DHCP_LAST_OPTION)
                {
#ifdef DHCPC_DEBUG
                logMsg ("file_is_opt:tag(%d) >= _DHCP_LAST_OPTION(%d)\n", tag, _DHCP_LAST_OPTION, 0, 0, 0, 0);
#endif
                optp++;
                optp += *optp;
                continue;
                }

            if (handle_param [ (unsigned int)tag] != NULL)
                {
                if ( (err = (*handle_param [ (unsigned int)tag]) (optp, parameter)) != 0)
                    return (err);
                else
                    SETBIT(parameter->got_option, tag);
                }

            /* Set the message type to distinguish DHCP and BOOTP messages. */

            else if (tag == _DHCP_MSGTYPE_TAG)
                SETBIT (parameter->got_option, tag);

            optp++;
            optp += *optp;
            }
        }
    else
        {
        if ( (parameter->file = calloc (1, strlen (msg->file) + 1)) == NULL)
            return (-1);
        strcpy (parameter->file, msg->file);
        }


    if (sname_is_opt)
        {
        endofopt = &msg->sname [MAX_SNAME];
        for (optp = msg->sname; optp <= endofopt; optp++)
            {
            tag = *optp;

            /* skip the PAD option */

            if (tag == _DHCP_PAD_TAG)
                continue;

            /* stop processing when the END option is reached */

            if (tag == _DHCP_END_TAG)
	        break;

            if (tag > _DHCP_LAST_OPTION)
                {
#ifdef DHCPC_DEBUG
                logMsg ("sname_is_opt:tag(%d) >= _DHCP_LAST_OPTION(%d)\n", tag, _DHCP_LAST_OPTION, 0, 0, 0, 0);
#endif
                optp++;
                optp += *optp;
                continue;
                }

            if (handle_param [ (unsigned int)tag] != NULL)
                {
	        if ( (err = (*handle_param [ (unsigned int)tag]) (optp, parameter)) != 0)
	            return(err);
                else
	            SETBIT (parameter->got_option, tag);
                }

            /* Set the message type to distinguish DHCP and BOOTP messages. */

            else if (tag == _DHCP_MSGTYPE_TAG)
                SETBIT (parameter->got_option, tag);

            optp++;
            optp += *optp;
            }
        }
    else
        {
        if ( (parameter->sname = calloc (1, strlen (msg->sname) + 1)) == NULL)
            return (-1);
        strcpy(parameter->sname, msg->sname);
        }

    parameter->ciaddr.s_addr = msg->ciaddr.s_addr;
    parameter->yiaddr.s_addr = msg->yiaddr.s_addr;
    parameter->siaddr.s_addr = msg->siaddr.s_addr;
    parameter->giaddr.s_addr = msg->giaddr.s_addr;

    if (ISSET (parameter->got_option, _DHCP_MSGTYPE_TAG))
        parameter->msgtype = DHCP_NATIVE;
    else
        parameter->msgtype = DHCP_BOOTP;

    /* Set lease duration to infinite for BOOTP replies. */

    if (parameter->msgtype == DHCP_BOOTP)
        {
        parameter->lease_duration = ~0;
        return (0);
        }

    /* Assign any server name provided if 'sname' used for options. */

    if (sname_is_opt)
        {
        parameter->sname = parameter->temp_sname;
        parameter->temp_sname = NULL;
        }

    /* Assign any bootfile provided if 'file' used for options. */

    if (file_is_opt)
        {
        if (ISSET (parameter->got_option, _DHCP_BOOTFILE_TAG))
            {
            parameter->file = parameter->temp_file;
            parameter->temp_file = NULL;
            }
        }

    if (parameter->dhcp_t1 == 0)
        {
        /* Timer t1 is half the lease duration - but don't divide. */

        parameter->dhcp_t1 = (parameter->lease_duration) >> 1;
        SETBIT (parameter->got_option, _DHCP_T1_TAG);
        }
    if (parameter->dhcp_t2 == 0)
        {
        /* Timer T2 is .875 of the lease - but don't use floating point. */

        int tmp = (parameter->lease_duration * 7) >> 3;
        parameter->dhcp_t2 = (unsigned short) tmp;
        SETBIT(parameter->got_option, _DHCP_T2_TAG);
        }

    return(0);
    }

/*******************************************************************************
*
* clean_param - clean the parameters data structure
*
* This routine frees all the memory allocated for the storage of parameters
* received from a DHCP server. It is called to remove the unselected offers
* or if an error occurs in the state machine after offers have been received.
*
* RETURNS: 0, always.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int clean_param
    (
    struct dhcp_param *param
    )
    {
    if (param == NULL)
        return(0);

    if (param->sname != NULL)
        free (param->sname);
    if (param->temp_sname != NULL)
        free (param->temp_sname);
    if (param->file != NULL)
        free (param->file);
    if (param->temp_file != NULL)
        free (param->temp_file);
    if (param->hostname != NULL)
        free (param->hostname);
    if (param->merit_dump != NULL)
        free (param->merit_dump);
    if (param->dns_domain != NULL)
        free (param->dns_domain);
    if (param->root_path != NULL)
        free (param->root_path);
    if (param->extensions_path != NULL)
        free (param->extensions_path);
    if (param->nis_domain != NULL)
        free (param->nis_domain);
    if (param->nb_scope != NULL)
        free (param->nb_scope);
    if (param->errmsg != NULL)
        free (param->errmsg);
    if (param->nisp_domain != NULL)
        free (param->nisp_domain);

    if (param->mtu_plateau_table != NULL)
        {
        if (param->mtu_plateau_table->shortnum != NULL)
            free (param->mtu_plateau_table->shortnum);
        free (param->mtu_plateau_table);
        }
    if (param->subnet_mask != NULL)
        free (param->subnet_mask);
    if (param->swap_server != NULL)
        free (param->swap_server);
    if (param->brdcast_addr != NULL)
        free (param->brdcast_addr);
    if (param->router != NULL)
        {
        if (param->router->addr != NULL)
            free (param->router->addr);
        free (param->router);
        }
    if (param->time_server != NULL)
        {
        if (param->time_server->addr != NULL)
            free (param->time_server->addr);
        free (param->time_server);
        }
    if (param->name_server != NULL)
        {
        if (param->name_server->addr != NULL)
            free (param->name_server->addr);
        free (param->name_server);
        }
    if (param->dns_server != NULL)
        {
        if (param->dns_server->addr != NULL)
            free (param->dns_server->addr);
        free (param->dns_server);
        }
    if (param->log_server != NULL)
        {
        if (param->log_server->addr != NULL)
            free (param->log_server->addr);
        free (param->log_server);
        }
    if (param->cookie_server != NULL)
        {
        if (param->cookie_server->addr != NULL)
            free (param->cookie_server->addr);
        free (param->cookie_server);
        }
    if (param->lpr_server != NULL)
        {
        if (param->lpr_server->addr != NULL)
            free (param->lpr_server->addr);
        free (param->lpr_server);
        }
    if (param->impress_server != NULL)
        {
        if (param->impress_server->addr != NULL)
            free (param->impress_server->addr);
        free (param->impress_server);
        }
    if (param->rls_server != NULL)
        {
        if (param->rls_server->addr != NULL)
            free (param->rls_server->addr);
        free (param->rls_server);
        }
    if (param->policy_filter != NULL)
        {
        if (param->policy_filter->addr != NULL)
            free (param->policy_filter->addr);
        free (param->policy_filter);
        }
    if (param->static_route != NULL)
        {
        if (param->static_route->addr != NULL)
            free (param->static_route->addr);
        free (param->static_route);
        }
    if (param->nis_server != NULL)
        {
        if (param->nis_server->addr != NULL)
            free (param->nis_server->addr);
        free (param->nis_server);
        }
    if (param->ntp_server != NULL)
        {
        if (param->ntp_server->addr != NULL)
            free (param->ntp_server->addr);
        free (param->ntp_server);
        }
    if (param->nbn_server != NULL)
        {
        if (param->nbn_server->addr != NULL)
            free (param->nbn_server->addr);
        free (param->nbn_server);
        }
    if (param->nbdd_server != NULL)
        {
        if (param->nbdd_server->addr != NULL)
            free (param->nbdd_server->addr);
        free (param->nbdd_server);
        }
    if (param->xfont_server != NULL)
        {
        if (param->xfont_server->addr != NULL)
            free (param->xfont_server->addr);
        free (param->xfont_server);
        }
    if (param->xdisplay_manager != NULL)
        {
        if (param->xdisplay_manager->addr != NULL)
            free (param->xdisplay_manager->addr);
        free (param->xdisplay_manager);
        }
    if (param->nisp_server != NULL)
        {
        if (param->nisp_server->addr != NULL)
            free (param->nisp_server->addr);
        free (param->nisp_server);
        }
    if (param->mobileip_ha != NULL)
        {
        if (param->mobileip_ha->addr != NULL)
            free (param->mobileip_ha->addr);
        free (param->mobileip_ha);
        }
    if (param->smtp_server != NULL)
        {
        if (param->smtp_server->addr != NULL)
            free (param->smtp_server->addr);
        free (param->smtp_server);
        }
    if (param->pop3_server != NULL)
        {
        if (param->pop3_server->addr != NULL)
            free (param->pop3_server->addr);
        free (param->pop3_server);
        }
    if (param->nntp_server != NULL)
        {
        if (param->nntp_server->addr != NULL)
            free (param->nntp_server->addr);
        free (param->nntp_server);
        }
    if (param->dflt_www_server != NULL)
        {
        if (param->dflt_www_server->addr != NULL)
            free (param->dflt_www_server->addr);
        free (param->dflt_www_server);
        }
    if (param->dflt_finger_server != NULL)
        {
        if (param->dflt_finger_server->addr != NULL)
            free (param->dflt_finger_server->addr);
        free (param->dflt_finger_server);
        }
    if (param->dflt_irc_server != NULL)
        {
        if (param->dflt_irc_server->addr != NULL)
            free (param->dflt_irc_server->addr);
        free (param->dflt_irc_server);
        }
    if (param->streettalk_server != NULL)
        {
        if (param->streettalk_server->addr != NULL)
            free (param->streettalk_server->addr);
        free (param->streettalk_server);
        }
    if (param->stda_server != NULL)
        {
        if (param->stda_server->addr != NULL)
            free (param->stda_server->addr);
        free (param->stda_server);
        }
    if (param->vendlist != NULL)
        free (param->vendlist);

    bzero ( (char *)param, sizeof (struct dhcp_param));
    return (0);
    }


/*******************************************************************************
*
* merge_param - combine the parameters from DHCP server responses
*
* This routine copies any parameters from an initial lease offer which were not
* supplied in the later acknowledgement from the server. If the acknowledgement
* from the server duplicates parameters from the offer, then the memory
* allocated for those parameters in the offer is released.
*
* RETURNS: 0, always.
*
* ERRNO: N/A
*
* NOMANUAL
*/

/*
 * if there is no information in newp but oldp, copy it to newp
 * else free the appropriate memory of oldp
 */
int merge_param
    (
    struct dhcp_param *oldp,     /* Parameters from lease offer. */
    struct dhcp_param *newp      /* Parameters from lease acknowledgement. */
    )
    {
    if (oldp == NULL || newp == NULL)
        return (0);

    if (newp->sname == NULL && oldp->sname != NULL)
        newp->sname = oldp->sname;
    else if (oldp->sname != NULL)
        free (oldp->sname);

    if (newp->file == NULL && oldp->file != NULL)
        newp->file = oldp->file;
    else if (oldp->file != NULL)
        free (oldp->file);

    if (newp->hostname == NULL && oldp->hostname != NULL)
        newp->hostname = oldp->hostname;
    else if (oldp->hostname != NULL)
        free (oldp->hostname);

    if (newp->merit_dump == NULL && oldp->merit_dump != NULL)
        newp->merit_dump = oldp->merit_dump;
    else if (oldp->merit_dump != NULL)
        free (oldp->merit_dump);

    if (newp->dns_domain == NULL && oldp->dns_domain != NULL)
        newp->dns_domain = oldp->dns_domain;
    else if (oldp->dns_domain != NULL)
        free (oldp->dns_domain);

    if (newp->root_path == NULL && oldp->root_path != NULL)
        newp->root_path = oldp->root_path;
    else if (oldp->root_path != NULL)
        free (oldp->root_path);

    if (newp->extensions_path == NULL && oldp->extensions_path != NULL)
        newp->extensions_path = oldp->extensions_path;
    else if (oldp->extensions_path != NULL)
        free (oldp->extensions_path);

    if (newp->nis_domain == NULL && oldp->nis_domain != NULL)
        newp->nis_domain = oldp->nis_domain;
    else if (oldp->nis_domain != NULL)
        free (oldp->nis_domain);

    if (newp->nb_scope == NULL && oldp->nb_scope != NULL)
        newp->nb_scope = oldp->nb_scope;
    else if (oldp->nb_scope != NULL)
        free (oldp->nb_scope);

    if (newp->errmsg == NULL && oldp->errmsg != NULL)
        newp->errmsg = oldp->errmsg;
    else if (oldp->errmsg != NULL)
        free (oldp->errmsg);

    if (newp->nisp_domain == NULL && oldp->nisp_domain != NULL)
        newp->nisp_domain = oldp->nisp_domain;
    else if (oldp->nisp_domain != NULL)
        free (oldp->nisp_domain);

    if (newp->mtu_plateau_table == NULL && oldp->mtu_plateau_table != NULL)
        newp->mtu_plateau_table = oldp->mtu_plateau_table;
    else
        {
        if (oldp->mtu_plateau_table != NULL)
            {
            if (oldp->mtu_plateau_table->shortnum != NULL)
                free (oldp->mtu_plateau_table->shortnum);
            free (oldp->mtu_plateau_table);
            }
        }

    if (newp->subnet_mask == NULL && oldp->subnet_mask != NULL)
        newp->subnet_mask = oldp->subnet_mask;
    else if (oldp->subnet_mask != NULL)
        free (oldp->subnet_mask);

    if (newp->swap_server == NULL && oldp->swap_server != NULL)
        newp->swap_server = oldp->swap_server;
    else if (oldp->swap_server != NULL)
        free (oldp->swap_server);

    if (newp->brdcast_addr == NULL && oldp->brdcast_addr != NULL)
        newp->brdcast_addr = oldp->brdcast_addr;
    else if (oldp->brdcast_addr != NULL)
        free (oldp->brdcast_addr);

    if (newp->router_solicit.s_addr == 0 && oldp->router_solicit.s_addr != 0)
        bcopy ( (char *)&oldp->router_solicit,
               (char *)&newp->router_solicit, sizeof (u_long));

    if (newp->router == NULL && oldp->router != NULL)
      newp->router = oldp->router;
    else
        {
        if (oldp->router != NULL && oldp->router->addr != NULL)
            free (oldp->router->addr);
        if (oldp->router != NULL)
            free (oldp->router);
        }

    if (newp->time_server == NULL && oldp->time_server != NULL)
        newp->time_server = oldp->time_server;
    else
        {
        if (oldp->time_server != NULL && oldp->time_server->addr != NULL)
            free (oldp->time_server->addr);
        if (oldp->time_server != NULL)
            free (oldp->time_server);
        }

    if (newp->name_server == NULL && oldp->name_server != NULL)
        newp->name_server = oldp->name_server;
    else
        {
        if (oldp->name_server != NULL && oldp->name_server->addr != NULL)
            free (oldp->name_server->addr);
        if (oldp->name_server != NULL)
            free (oldp->name_server);
        }

    if (newp->dns_server == NULL && oldp->dns_server != NULL)
        newp->dns_server = oldp->dns_server;
    else
        {
        if (oldp->dns_server != NULL && oldp->dns_server->addr != NULL)
            free (oldp->dns_server->addr);
        if (oldp->dns_server != NULL)
            free (oldp->dns_server);
        }

    if (newp->log_server == NULL && oldp->log_server != NULL)
        newp->log_server = oldp->log_server;
    else
        {
        if (oldp->log_server != NULL && oldp->log_server->addr != NULL)
            free (oldp->log_server->addr);
        if (oldp->log_server != NULL)
            free (oldp->log_server);
        }

    if (newp->cookie_server == NULL && oldp->cookie_server != NULL)
        newp->cookie_server = oldp->cookie_server;
    else
        {
        if (oldp->cookie_server != NULL && oldp->cookie_server->addr != NULL)
            free (oldp->cookie_server->addr);
        if (oldp->cookie_server != NULL)
            free (oldp->cookie_server);
        }

    if (newp->lpr_server == NULL && oldp->lpr_server != NULL)
        newp->lpr_server = oldp->lpr_server;
    else
        {
        if (oldp->lpr_server != NULL && oldp->lpr_server->addr != NULL)
            free (oldp->lpr_server->addr);
        if (oldp->lpr_server != NULL)
            free (oldp->lpr_server);
        }

    if (newp->impress_server == NULL && oldp->impress_server != NULL)
        newp->impress_server = oldp->impress_server;
    else
        {
        if (oldp->impress_server != NULL && oldp->impress_server->addr != NULL)
            free (oldp->impress_server->addr);
        if (oldp->impress_server != NULL)
            free (oldp->impress_server);
        }

    if (newp->rls_server == NULL && oldp->rls_server != NULL)
        newp->rls_server = oldp->rls_server;
    else
        {
        if (oldp->rls_server != NULL && oldp->rls_server->addr != NULL)
            free (oldp->rls_server->addr);
        if (oldp->rls_server != NULL)
            free (oldp->rls_server);
        }

    if (newp->policy_filter == NULL && oldp->policy_filter != NULL)
        newp->policy_filter = oldp->policy_filter;
    else
        {
        if (oldp->policy_filter != NULL && oldp->policy_filter->addr != NULL)
            free (oldp->policy_filter->addr);
        if (oldp->policy_filter != NULL)
            free (oldp->policy_filter);
        }

    if (newp->static_route == NULL && oldp->static_route != NULL)
        newp->static_route = oldp->static_route;
    else
        {
        if (oldp->static_route != NULL && oldp->static_route->addr != NULL)
            free (oldp->static_route->addr);
        if (oldp->static_route != NULL)
            free (oldp->static_route);
        }

    if (newp->nis_server == NULL && oldp->nis_server != NULL)
        newp->nis_server = oldp->nis_server;
    else
        {
        if (oldp->nis_server != NULL && oldp->nis_server->addr != NULL)
            free (oldp->nis_server->addr);
        if (oldp->nis_server != NULL)
            free (oldp->nis_server);
        }

    if (newp->ntp_server == NULL && oldp->ntp_server != NULL)
        newp->ntp_server = oldp->ntp_server;
    else
        {
        if (oldp->ntp_server != NULL && oldp->ntp_server->addr != NULL)
            free (oldp->ntp_server->addr);
        if (oldp->ntp_server != NULL)
            free (oldp->ntp_server);
        }

    if (newp->nbn_server == NULL && oldp->nbn_server != NULL)
        newp->nbn_server = oldp->nbn_server;
    else
        {
        if (oldp->nbn_server != NULL && oldp->nbn_server->addr != NULL)
            free (oldp->nbn_server->addr);
        if (oldp->nbn_server != NULL)
            free (oldp->nbn_server);
        }

    if (newp->nbdd_server == NULL && oldp->nbdd_server != NULL)
        newp->nbdd_server = oldp->nbdd_server;
    else
        {
        if (oldp->nbdd_server != NULL && oldp->nbdd_server->addr != NULL)
            free (oldp->nbdd_server->addr);
        if (oldp->nbdd_server != NULL)
            free (oldp->nbdd_server);
        }

    if (newp->xfont_server == NULL && oldp->xfont_server != NULL)
        newp->xfont_server = oldp->xfont_server;
    else
        {
        if (oldp->xfont_server != NULL && oldp->xfont_server->addr != NULL)
            free (oldp->xfont_server->addr);
        if (oldp->xfont_server != NULL)
            free (oldp->xfont_server);
        }

    if (newp->xdisplay_manager == NULL && oldp->xdisplay_manager != NULL)
        newp->xdisplay_manager = oldp->xdisplay_manager;
    else
        {
        if (oldp->xdisplay_manager != NULL &&
            oldp->xdisplay_manager->addr != NULL)
            free (oldp->xdisplay_manager->addr);
        if (oldp->xdisplay_manager != NULL)
            free (oldp->xdisplay_manager);
        }

    if (newp->nisp_server == NULL && oldp->nisp_server != NULL)
        newp->nisp_server = oldp->nisp_server;
    else
        {
        if (oldp->nisp_server != NULL && oldp->nisp_server->addr != NULL)
            free (oldp->nisp_server->addr);
        if (oldp->nisp_server != NULL)
            free (oldp->nisp_server);
        }

    if (newp->mobileip_ha == NULL && oldp->mobileip_ha != NULL)
        newp->mobileip_ha = oldp->mobileip_ha;
    else
        {
        if (oldp->mobileip_ha != NULL && oldp->mobileip_ha->addr != NULL)
            free (oldp->mobileip_ha->addr);
        if (oldp->mobileip_ha != NULL)
            free (oldp->mobileip_ha);
        }

    if (newp->smtp_server == NULL && oldp->smtp_server != NULL)
        newp->smtp_server = oldp->smtp_server;
    else
        {
        if (oldp->smtp_server != NULL && oldp->smtp_server->addr != NULL)
            free (oldp->smtp_server->addr);
        if (oldp->smtp_server != NULL)
            free (oldp->smtp_server);
        }

    if (newp->pop3_server == NULL && oldp->pop3_server != NULL)
        newp->pop3_server = oldp->pop3_server;
    else
        {
        if (oldp->pop3_server != NULL && oldp->pop3_server->addr != NULL)
            free (oldp->pop3_server->addr);
        if (oldp->pop3_server != NULL)
            free (oldp->pop3_server);
        }

    if (newp->nntp_server == NULL && oldp->nntp_server != NULL)
        newp->nntp_server = oldp->nntp_server;
    else
        {
        if (oldp->nntp_server != NULL && oldp->nntp_server->addr != NULL)
            free (oldp->nntp_server->addr);
        if (oldp->nntp_server != NULL)
            free (oldp->nntp_server);
        }

    if (newp->dflt_www_server == NULL && oldp->dflt_www_server != NULL)
        newp->dflt_www_server = oldp->dflt_www_server;
    else
        {
        if (oldp->dflt_www_server != NULL &&
            oldp->dflt_www_server->addr != NULL)
            free (oldp->dflt_www_server->addr);
        if (oldp->dflt_www_server != NULL)
            free (oldp->dflt_www_server);
        }

    if (newp->dflt_finger_server == NULL && oldp->dflt_finger_server != NULL)
        newp->dflt_finger_server = oldp->dflt_finger_server;
    else
        {
        if (oldp->dflt_finger_server != NULL &&
            oldp->dflt_finger_server->addr != NULL)
            free (oldp->dflt_finger_server->addr);
        if (oldp->dflt_finger_server != NULL)
            free (oldp->dflt_finger_server);
        }

    if (newp->dflt_irc_server == NULL && oldp->dflt_irc_server != NULL)
        newp->dflt_irc_server = oldp->dflt_irc_server;
    else
        {
        if (oldp->dflt_irc_server != NULL &&
            oldp->dflt_irc_server->addr != NULL)
            free (oldp->dflt_irc_server->addr);
        if (oldp->dflt_irc_server != NULL)
            free (oldp->dflt_irc_server);
        }

    if (newp->streettalk_server == NULL && oldp->streettalk_server != NULL)
        newp->streettalk_server = oldp->streettalk_server;
    else
        {
        if (oldp->streettalk_server != NULL &&
            oldp->streettalk_server->addr != NULL)
            free (oldp->streettalk_server->addr);
        if (oldp->streettalk_server != NULL)
            free (oldp->streettalk_server);
        }

    if (newp->stda_server == NULL && oldp->stda_server != NULL)
        newp->stda_server = oldp->stda_server;
    else
        {
        if (oldp->stda_server != NULL && oldp->stda_server->addr != NULL)
            free (oldp->stda_server->addr);
        if (oldp->stda_server != NULL)
            free (oldp->stda_server);
        }

    /* Remove any vendor-specific information from an earlier response. */

    if (oldp->vendlist != NULL)
        free (oldp->vendlist);

    return (0);
    }


/*******************************************************************************
*
* initialize - initialize data structures for network message transfer
*
* This routine sets the DHCP ports, resets the network interface, and allocates
* storage for incoming messages. It is called from the dhcp_client_setup()
* routine during the overall initialization of the DHCP client library.
*
* RETURNS: 0 if initialization successful, or -1 otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int initialize
    (
    int 	serverPort, 	/* port monitored by DHCP servers */
    int 	clientPort 	/* port monitored by DHCP client */
    )
    {
    char * 	sbufp;

    /* Always use default ports for client and server. */

    dhcps_port = htons (serverPort);
    dhcpc_port = htons (clientPort);

    dhcpif.rbufsize = DFLTDHCPLEN + UDPHL + IPHL + ETHERHL;

    /* Include offset to allow 4-byte alignment of IP header for Sun BSP's. */

    dhcpif.rbuf = (char *)memalign (4, dhcpif.rbufsize + DHCPC_OFF);
    if (dhcpif.rbuf == NULL)
        {
#ifdef DHCPC_DEBUG
        logMsg ("allocation error for rbuf in initialize\n", 0, 0, 0, 0, 0, 0);
#endif
        return (-1);
        }
    bzero (dhcpif.rbuf, dhcpif.rbufsize + DHCPC_OFF);

    pDhcpcRecvBuf = (char *)memalign (4, dhcpif.rbufsize + DHCPC_OFF);
    if (pDhcpcRecvBuf == NULL)
        {
        return (-1);
        }
    bzero (pDhcpcRecvBuf, dhcpif.rbufsize + DHCPC_OFF);

    sbuf.size = ETHERHL + IPHL + UDPHL + DFLTDHCPLEN;

    /* Include offset to allow 4-byte alignment of IP header for Sun BSP's. */

    if ( (sbuf.buf = (char *)memalign (4, sbuf.size + DHCPC_OFF)) == NULL)
        {
#ifdef DHCPC_DEBUG
        logMsg ("allocation error for sbuf in initialize\n", 0, 0, 0, 0, 0, 0);
#endif
        return(-1);
        }
    bzero (sbuf.buf, sbuf.size + DHCPC_OFF);

    sbufp = &sbuf.buf [DHCPC_OFF];
    dhcpcMsgOut.ether = (struct ether_header *)sbufp;
    dhcpcMsgOut.ip = (struct ip *)&sbufp [ETHERHL];
    dhcpcMsgOut.udp = (struct udphdr *)&sbufp [ETHERHL + IPHL];
    dhcpcMsgOut.dhcp = (struct dhcp *)&sbufp [ETHERHL + IPHL + UDPHL];

    return (0);
    }

/*******************************************************************************
*
* reset_if - halt the network and reset the network interface
*
* This routine sets the IP address of the network interface to a random
* value of 10.x.x.x, which is an old ARPA debugging address, resets the
* broadcast address and subnet mask, and flushes the routing tables.
* It is called before initiating a lease negotiation if no event notification
* hook is present. It is also always called for a lease established at boot
* time, whether or not an event hook is registered for that lease.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void reset_if
    (
    struct if_info * 	pIfData 	/* interface used by lease */
    )
    {
    struct in_addr addr;
    struct in_addr mask;
    struct in_addr brdaddr;

    addr.s_addr = htonl ( (generate_xid (pIfData) & 0xfff) | 0x0a000000);

    mask.s_addr = htonl (0xff000000);
    brdaddr.s_addr = inet_addr ("255.255.255.255");

    config_if (pIfData, &addr, &mask, &brdaddr);

    flushroutes ();
    }

/*******************************************************************************
*
* down_if - change network interface flags
*
* This routine clears the IFF_UP flag of the network interface. It is called
* when a lease is manually terminated with a dhcpcRelease() or dhcpcShutdown()
* call if no event hook is present for the lease. It is always called when
* removing a lease established at boot time.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void down_if
    (
    struct if_info *ifp
    )
    {
    int sockfd;
    struct ifreq ridreq;

    reset_if (ifp);

    if ( (sockfd = socket (AF_INET, SOCK_RAW, 0)) < 0)
        return;

    bzero ( (char *)&ridreq, sizeof (struct ifreq));
    sprintf (ridreq.ifr_name, "%s%d", ifp->name, ifp->unit);
    ridreq.ifr_addr.sa_len = sizeof (struct sockaddr_in);

    /* down interface */
    ioctl (sockfd, SIOCGIFFLAGS, (int) (caddr_t) &ridreq);
    ridreq.ifr_flags &= (~IFF_UP);
    ioctl (sockfd, SIOCSIFFLAGS, (int) (caddr_t) &ridreq);

    close (sockfd);

    return;
    }

/*******************************************************************************
*
* config_if - configure network interface
*
* This routine sets one or more of the current address, broadcast address, and
* subnet mask, depending on whether or not the corresponding parameters are
* NULL.  It is called when a lease is obtained or the network is reset. When a
* new lease is obtained, a successful return value is used to cause assignment
* of new routing tables.
*
* RETURNS: -1 on error, 0 if new settings assigned, 1 otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int config_if
    (
    struct if_info *ifp,
    struct in_addr *addr,
    struct in_addr *mask,
    struct in_addr *brdcst
    )
    {
    int sockfd = 0;
    int status;
    struct ifreq ifr;
    struct in_addr current_addr;
    struct in_addr  current_mask;
    struct in_addr  current_brdcst;

    if ( (sockfd = socket (AF_INET, SOCK_RAW, 0)) < 0)
        return (-1);

    bzero ( (char *)&current_addr, sizeof (current_addr));
    bzero ( (char *)&current_mask, sizeof (current_mask));
    bzero ( (char *)&current_brdcst, sizeof (current_brdcst));
    bzero ( (char *)&ifr, sizeof (struct ifreq));

    sprintf (ifr.ifr_name, "%s%d", ifp->name, ifp->unit);
    ifr.ifr_addr.sa_len = sizeof (struct sockaddr_in);

    status = ioctl (sockfd, SIOCGIFADDR, (int)&ifr);
    current_addr.s_addr =
                       ( (struct sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr;

    status = ioctl (sockfd, SIOCGIFNETMASK, (int)&ifr);
    current_mask.s_addr =
                       ( (struct sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr;

    status = ioctl (sockfd, SIOCGIFBRDADDR, (int)&ifr);
    current_brdcst.s_addr =
                  ( (struct sockaddr_in *)&ifr.ifr_broadaddr)->sin_addr.s_addr;

    if (current_addr.s_addr == addr->s_addr &&
        (mask == NULL || current_mask.s_addr == mask->s_addr) &&
        (brdcst == NULL || current_brdcst.s_addr == brdcst->s_addr))
        {
        close (sockfd);
        return (1);
        }

    flushroutes ();

    /* down interface */

    ioctl (sockfd, SIOCGIFFLAGS, (int)&ifr);
    ifr.ifr_flags &= (~IFF_UP);
    ioctl (sockfd, SIOCSIFFLAGS, (int)&ifr);

    /*
     * Deleting the interface address is required to correctly scrub the
     * routing table based on the current netmask.
     */

    bzero ( (char *)&ifr, sizeof (struct ifreq));
    sprintf (ifr.ifr_name, "%s%d", ifp->name, ifp->unit);
    ifr.ifr_addr.sa_len = sizeof (struct sockaddr_in);
    ( (struct sockaddr_in *) &ifr.ifr_addr)->sin_family = AF_INET;
    ( (struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr =
                                                           current_addr.s_addr;
    status = ioctl (sockfd, SIOCDIFADDR, (int)&ifr);
    if (status < 0)
        {
         /* Sometimes no address has been set, so ignore that error. */

        if (errno != EADDRNOTAVAIL)
            {
            close (sockfd);
            return (-1);
            }
        }

    if (mask != NULL)
        {
        bzero ( (char *)&ifr, sizeof (struct ifreq));
        sprintf (ifr.ifr_name, "%s%d", ifp->name, ifp->unit);
        ifr.ifr_addr.sa_len = sizeof (struct sockaddr_in);
        ( (struct sockaddr_in *) &ifr.ifr_addr)->sin_family = AF_INET;
        ( (struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr =
                                                                 mask->s_addr;
        status = ioctl (sockfd, SIOCSIFNETMASK, (int)&ifr);
        if (status < 0)
            {
            close (sockfd);
            return (-1);
            }
        }

    if (brdcst != NULL)
        {
        bzero ( (char *)&ifr, sizeof (struct ifreq));
        sprintf (ifr.ifr_name, "%s%d", ifp->name, ifp->unit);
        ifr.ifr_addr.sa_len = sizeof (struct sockaddr_in);
        ( (struct sockaddr_in *)&ifr.ifr_broadaddr)->sin_family = AF_INET;
        ( (struct sockaddr_in *)&ifr.ifr_broadaddr)->sin_addr.s_addr =
                                                               brdcst->s_addr;
        status = ioctl (sockfd, SIOCSIFBRDADDR, (int)&ifr);
        if (status < 0)
            {
            close (sockfd);
            return (-1);
            }
        }

    if (addr != NULL)
        {
        bzero ( (char *)&ifr, sizeof (struct ifreq));
        sprintf (ifr.ifr_name, "%s%d", ifp->name, ifp->unit);
        ifr.ifr_addr.sa_len = sizeof (struct sockaddr_in);
        ( (struct sockaddr_in *) &ifr.ifr_addr)->sin_family = AF_INET;
        ( (struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr =
                                                                 addr->s_addr;
        status = ioctl (sockfd, SIOCSIFADDR, (int)&ifr);
        if (status < 0)
            {
            close (sockfd);
            return (-1);
            }
        }

    /* bring interface back up */

    ioctl (sockfd, SIOCGIFFLAGS, (int)&ifr);
    ifr.ifr_flags |= IFF_UP;
    ioctl (sockfd, SIOCSIFFLAGS, (int)&ifr);
    close (sockfd);

    return (0);
    }

/*******************************************************************************
*
* set_route - set network routing table
*
* This routine is called when config_if() assigns new address information.
* It sets the default route for a new lease if the DHCP server provided the
* router IP address.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void set_route
    (
    struct dhcp_param *param
    )
    {
    int sockfd = 0;

#if BSD<44
    struct rtentry rt;
#else
    struct ortentry rt;
#endif

    struct sockaddr dst, gateway;

    if (param == NULL)
        return;

    sockfd = socket(AF_INET, SOCK_RAW, 0);
    if (sockfd < 0)
        {
#ifdef DHCPC_DEBUG
        logMsg("socket() error in set_route()\n", 0, 0, 0, 0, 0, 0);
#endif
        return;
        }

    /* set default route, if router IP address is available. */

    if (ISSET(param->got_option, _DHCP_ROUTER_TAG) && param->router != NULL &&
        param->router->addr != NULL)
        {
#if BSD<44
        bzero ( (char *)&rt, sizeof (struct rtentry));
#else
        bzero ( (char *)&rt, sizeof (struct ortentry));
#endif

        bzero ( (char *)&dst, sizeof (struct sockaddr));
        bzero ( (char *)&gateway, sizeof (struct sockaddr));
        rt.rt_flags = RTF_UP | RTF_GATEWAY;

        ( (struct sockaddr_in *)&dst)->sin_family = AF_INET;
        ( (struct sockaddr_in *)&dst)->sin_len = sizeof (struct sockaddr_in);
        ( (struct sockaddr_in *)&dst)->sin_addr.s_addr = INADDR_ANY;

        ( (struct sockaddr_in *)&gateway)->sin_family = AF_INET;
        ( (struct sockaddr_in *)&gateway)->sin_len =
                                                   sizeof (struct sockaddr_in);
        ( (struct sockaddr_in *)&gateway)->sin_addr.s_addr =
                                                   param->router->addr->s_addr;
        rt.rt_dst = dst;
        rt.rt_gateway = gateway;

        if (ioctl (sockfd, SIOCADDRT, (int)&rt) < 0)
            {
#ifdef DHCPC_DEBUG
            logMsg ("SIOCADDRT (default route)\n", 0, 0, 0, 0, 0, 0);
#endif
            close (sockfd);
            }
        }

    close (sockfd);
    return;
    }

/*******************************************************************************
*
* make_decline - construct a DHCP decline message
*
* This routine constructs an outgoing UDP/IP message containing the values
* required to decline an offered IP address.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void make_decline
    (
    char *lsbuf,
    struct dhcp_reqspec *pDhcpcReqSpec,
    struct if_info * 	pIfData 	/* interface used by lease */
    )
    {
    int offopt = 0;    /* offset in options field */
    int i = 0;
    u_long tmpul = 0;
    struct dhcp    *lsdhcp = NULL;
    struct udphdr  *lsudp = NULL;
    struct ip      *lsip = NULL;
    struct ether_header *lsether = NULL;
    struct ps_udph pudph;

    bzero ( (char *)&pudph, sizeof (pudph));

    lsether = (struct ether_header *)lsbuf;
    lsip = (struct ip *)&lsbuf [ETHERHL];
    lsudp = (struct udphdr *)&lsbuf [ETHERHL + IPHL];
    lsdhcp = (struct dhcp *)&lsbuf [ETHERHL + IPHL + UDPHL];

    /* construct dhcp part */

    lsdhcp->op = BOOTREQUEST;
    lsdhcp->htype = pIfData->haddr.htype;
    lsdhcp->hlen = pIfData->haddr.hlen;
    lsdhcp->xid = generate_xid (pIfData);
    lsdhcp->giaddr = dhcpcMsgIn.dhcp->giaddr;
    bcopy (pIfData->haddr.haddr, lsdhcp->chaddr, lsdhcp->hlen);

    /* insert magic cookie */

    bcopy ( (char *)dhcpCookie, lsdhcp->options, MAGIC_LEN);
    offopt = MAGIC_LEN;

    /* insert message type */

    lsdhcp->options [offopt++] = _DHCP_MSGTYPE_TAG;
    lsdhcp->options [offopt++] = 1;
    lsdhcp->options [offopt++] = DHCPDECLINE;

    /* insert requested IP */

    if (pDhcpcReqSpec->ipaddr.s_addr == 0)
        return;
    else if (CHKOFF (4))
        {
        lsdhcp->options [offopt++] = _DHCP_REQUEST_IPADDR_TAG;
        lsdhcp->options [offopt++] = 4;
        bcopy ( (char *)&pDhcpcReqSpec->ipaddr, &lsdhcp->options [offopt], 4);
        offopt += 4;
        }

    /* insert client identifier */

    if (pDhcpcReqSpec->clid != NULL && CHKOFF (pDhcpcReqSpec->clid->len + 1))
        {
        lsdhcp->options [offopt++] = _DHCP_CLIENT_ID_TAG;
        lsdhcp->options [offopt++] = pDhcpcReqSpec->clid->len + 1;
        lsdhcp->options [offopt++] = pDhcpcReqSpec->clid->type;
        bcopy (pDhcpcReqSpec->clid->id, &lsdhcp->options [offopt],
               pDhcpcReqSpec->clid->len);
        offopt += pDhcpcReqSpec->clid->len;
        }

    /* insert server identifier */

    if (CHKOFF (4))
       {
       lsdhcp->options [offopt++] = _DHCP_SERVER_ID_TAG;
       lsdhcp->options [offopt++] = 4;
       bcopy ( (char *)&pDhcpcReqSpec->srvaddr, &lsdhcp->options [offopt], 4);
       offopt += 4;
       }

    /* if necessary, insert error message */

    if (pDhcpcReqSpec->dhcp_errmsg != NULL &&
        CHKOFF (strlen (pDhcpcReqSpec->dhcp_errmsg)))
        {
        lsdhcp->options [offopt++] = _DHCP_ERRMSG_TAG;
        lsdhcp->options [offopt++] = strlen (pDhcpcReqSpec->dhcp_errmsg);
        bcopy (pDhcpcReqSpec->dhcp_errmsg, &lsdhcp->options [offopt],
        strlen (pDhcpcReqSpec->dhcp_errmsg));
        offopt += strlen (pDhcpcReqSpec->dhcp_errmsg);
        }

    /* if necessary, insert host name */
    if (pDhcpcReqSpec->hostName != NULL &&
        CHKOFF (strlen (pDhcpcReqSpec->hostName)))
        {
        lsdhcp->options [offopt++] = _DHCP_HOSTNAME_TAG;
        lsdhcp->options [offopt++] = strlen (pDhcpcReqSpec->hostName);
        bcopy (pDhcpcReqSpec->hostName, &lsdhcp->options [offopt],
        strlen (pDhcpcReqSpec->hostName));
        offopt += strlen (pDhcpcReqSpec->hostName);
        }

    lsdhcp->options [offopt] = _DHCP_END_TAG;

    /* construct udp part */

    lsudp->uh_sport = dhcpc_port;
    lsudp->uh_dport = dhcps_port;
    lsudp->uh_ulen = htons (DFLTDHCPLEN + UDPHL);
    lsudp->uh_sum = 0;

    /* fill pseudo udp header */

    pudph.srcip.s_addr = 0;
    pudph.dstip.s_addr = pDhcpcReqSpec->srvaddr.s_addr;
    pudph.zero = 0;
    pudph.prto = IPPROTO_UDP;
    pudph.ulen = lsudp->uh_ulen;
    lsudp->uh_sum = udp_cksum(&pudph, (char *) lsudp, ntohs(pudph.ulen));

    /* construct ip part */

#if BSD<44
    lsip->ip_v_hl = 0;
    lsip->ip_v_hl = IPVERSION << 4;
    lsip->ip_v_hl |= IPHL >> 2;
#else
    lsip->ip_v = IPVERSION;
    lsip->ip_hl = IPHL >> 2;
#endif
    lsip->ip_tos = 0;
    lsip->ip_len = htons(DFLTDHCPLEN + UDPHL + IPHL);
    tmpul = generate_xid (pIfData);
    tmpul += (tmpul >> 16);
    lsip->ip_id = (u_short) (~tmpul);
    lsip->ip_off = htons(IP_DF);                         /* XXX */
    lsip->ip_ttl = 0x20;                                 /* XXX */
    lsip->ip_p = IPPROTO_UDP;
    lsip->ip_src.s_addr = 0;
    lsip->ip_dst.s_addr = dhcpcMsgIn.ip->ip_src.s_addr;
    lsip->ip_sum = 0;

#if BSD<44
    lsip->ip_sum = checksum((u_short *)lsip, (lsip->ip_v_hl & 0xf) << 2);
#else
    lsip->ip_sum = checksum((u_short *)lsip, lsip->ip_hl << 2);
#endif

    /* construct ether part */

    for (i = 0; i < 6; i++)
        {
        lsether->ether_dhost[i] = dhcpcMsgIn.ether->ether_shost[i];
        lsether->ether_shost[i] = pIfData->haddr.haddr[i];
        }
    lsether->ether_type = ETHERTYPE_IP;

    return;
    }

/*******************************************************************************
*
* make_release - construct a DHCP release message
*
* This routine constructs an outgoing UDP/IP message containing the values
* required to relinquish the active lease.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void make_release
    (
    struct dhcp *lsdhcp,
    struct dhcp_reqspec *pDhcpcReqSpec,
    struct if_info * 	pIfData 	/* interface used by lease */
    )
    {
    int offopt = 0;    /* offset in options field */

    /* construct dhcp part */

    lsdhcp->op = BOOTREQUEST;
    lsdhcp->htype = pIfData->haddr.htype;
    lsdhcp->hlen = pIfData->haddr.hlen;
    lsdhcp->xid = generate_xid (pIfData);
    lsdhcp->ciaddr = pDhcpcReqSpec->ipaddr;
    bcopy (pIfData->haddr.haddr, lsdhcp->chaddr, lsdhcp->hlen);

    /* insert magic cookie */

    bcopy ( (char *)dhcpCookie, lsdhcp->options, MAGIC_LEN);
    offopt = MAGIC_LEN;

    /* insert message type */

    lsdhcp->options [offopt++] = _DHCP_MSGTYPE_TAG;
    lsdhcp->options [offopt++] = 1;
    lsdhcp->options [offopt++] = DHCPRELEASE;

    /* insert server identifier */

    if (CHKOFF (4))
        {
        lsdhcp->options [offopt++] = _DHCP_SERVER_ID_TAG;
        lsdhcp->options [offopt++] = 4;
        bcopy ( (char *)&pDhcpcReqSpec->srvaddr, &lsdhcp->options [offopt], 4);
        offopt += 4;
        }

    /* insert client identifier */

    if (pDhcpcReqSpec->clid != NULL && CHKOFF (pDhcpcReqSpec->clid->len + 1))
        {
        lsdhcp->options [offopt++] = _DHCP_CLIENT_ID_TAG;
        lsdhcp->options [offopt++] = pDhcpcReqSpec->clid->len + 1;
        lsdhcp->options [offopt++] = pDhcpcReqSpec->clid->type;
        bcopy (pDhcpcReqSpec->clid->id, &lsdhcp->options [offopt],
               pDhcpcReqSpec->clid->len);
        offopt += pDhcpcReqSpec->clid->len;
        }

    /* if necessary, insert error message */

    if (pDhcpcReqSpec->dhcp_errmsg != NULL &&
        CHKOFF (strlen (pDhcpcReqSpec->dhcp_errmsg)))
        {
        lsdhcp->options [offopt++] = _DHCP_ERRMSG_TAG;
        lsdhcp->options [offopt++] = strlen (pDhcpcReqSpec->dhcp_errmsg);
        bcopy(pDhcpcReqSpec->dhcp_errmsg, &lsdhcp->options [offopt],
              strlen (pDhcpcReqSpec->dhcp_errmsg));
        offopt += strlen (pDhcpcReqSpec->dhcp_errmsg);
        }

    /* if necessary, insert host name */

    if (pDhcpcReqSpec->hostName != NULL &&
        CHKOFF (strlen (pDhcpcReqSpec->hostName)))
        {
        lsdhcp->options [offopt++] = _DHCP_HOSTNAME_TAG;
        lsdhcp->options [offopt++] = strlen (pDhcpcReqSpec->hostName);
        bcopy (pDhcpcReqSpec->hostName, &lsdhcp->options [offopt],
        strlen (pDhcpcReqSpec->hostName));
        offopt += strlen (pDhcpcReqSpec->hostName);
        }

    lsdhcp->options [offopt] = _DHCP_END_TAG;

    return;
    }

/*******************************************************************************
*
* dhcp_decline - send a DHCP decline message
*
* This routine constructs a message declining an offered IP address and sends
* it directly to the responding server. It is called when an ARP request
* detects that the offered address is already in use.
*
* RETURNS: 0 if message sent, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int dhcp_decline
    (
    struct dhcp_reqspec * 	pDhcpcReqSpec,
    struct if_info * 		pIfData 	/* interface used by lease */
    )
    {
    int  lsbufsize = ETHERHL + IPHL + UDPHL + DFLTDHCPLEN;
    char *lsbuf;
    char *lsbufp;

#ifdef DHCPC_DEBUG
    char output [INET_ADDR_LEN];
#endif

    if (pDhcpcReqSpec->srvaddr.s_addr == 0)
        return(-1);

    /*
     * Allocate clean memory and build new DHCP message. Use offset to
     * provide 4-byte alignment of IP header needed by Sun BSP's.
     */

    lsbuf = (char *)memalign (4, DHCPC_OFF + lsbufsize);
    if (lsbuf == NULL)
        return (-1);
    bzero (lsbuf, lsbufsize + DHCPC_OFF);

    lsbufp = &lsbuf [DHCPC_OFF];
    bzero (lsbufp, lsbufsize);
    make_decline (lsbufp, pDhcpcReqSpec, pIfData);

    if (etherOutput (pIfData->iface, (struct ether_header *)lsbufp,
                     &lsbufp [ETHERHL], lsbufsize - ETHERHL) == ERROR)
        {
        free (lsbuf);
        return (-1);
        }

#ifdef DHCPC_DEBUG
    inet_ntoa_b (pDhcpcReqSpec->ipaddr, output);
    logMsg ("send DHCPDECLINE(%s)\n", (int)output, 0, 0, 0, 0, 0);
#endif
    free (lsbuf);
    return (0);
    }

/*******************************************************************************
*
* dhcp_release - send a DHCP release message
*
* This routine constructs a message declining an offered IP address and sends
* it directly to the responding server. It is called when an error prevents
* the use of an acquired lease, or when the lease is relinquished manually
* by a dhcpcRelease() or dhcpcShutdown() call. The message is sent directly to
* the responding DHCP server.
*
* RETURNS: 0 if message sent, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int dhcp_release
    (
    struct dhcp_reqspec *pDhcpcReqSpec,
    struct if_info * 		pIfData 	/* interface used by lease */
    )
    {
    int  lsbufsize = DFLTDHCPLEN;
    char lsbuf [DFLTDHCPLEN];
    struct dhcp *lsdhcp = NULL;
#ifdef DHCPC_DEBUG
    char output [INET_ADDR_LEN];
#endif

    bzero (lsbuf, lsbufsize);

    lsdhcp = (struct dhcp *)lsbuf;
    if (pDhcpcReqSpec->srvaddr.s_addr == 0)
        return (-1);

    /* send DHCP message */

    make_release (lsdhcp, pDhcpcReqSpec, pIfData);

    if (send_unicast (&pDhcpcReqSpec->srvaddr, lsdhcp) < 0)
        return (-1);

#ifdef DHCPC_DEBUG
    inet_ntoa_b (pDhcpcReqSpec->ipaddr, output);
    logMsg("send DHCPRELEASE(%s)\n", (int)output, 0, 0, 0, 0, 0);
#endif
    return(0);
    }

/*******************************************************************************
*
* set_declinfo - initialize request specification for decline message
*
* This routine assigns the fields in the request specifier used to construct
* messages to the appropriate values for a DHCP decline message according to
* the parameters of the currently active lease.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void set_declinfo
    (
    struct dhcp_reqspec *pDhcpcReqSpec,
    LEASE_DATA * 	pLeaseData,
    char *errmsg,
    int arpans
    )
    {
    char output [INET_ADDR_LEN];
    struct dhcp_param * 	paramp;

    paramp = pLeaseData->dhcpcParam;

    pDhcpcReqSpec->ipaddr = paramp->yiaddr;
    pDhcpcReqSpec->srvaddr = paramp->server_id;
    if (pLeaseData->leaseReqSpec.clid != NULL)
        pDhcpcReqSpec->clid = pLeaseData->leaseReqSpec.clid;
    else
        pDhcpcReqSpec->clid = NULL;
    if (errmsg[0] == 0)
        {
        inet_ntoa_b (paramp->yiaddr, output);
        if (arpans != OK)
            sprintf (errmsg, "IP address (%s) is already in use.", output);
        else
            sprintf (errmsg, "IP address (%s) doesn't match requested value.",
	                      output);
        }
    pDhcpcReqSpec->dhcp_errmsg = errmsg;

    return;
    }

/*******************************************************************************
*
* set_relinfo - initialize request specification for release message
*
* This routine assigns the fields in the request specifier used to construct
* messages to the appropriate values for a DHCP release message according to
* the parameters of the currently active lease.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void set_relinfo
    (
    struct dhcp_reqspec *pDhcpcReqSpec,
    LEASE_DATA * 	pLeaseData,
    char *errmsg
    )
    {
    char output [INET_ADDR_LEN];
    struct dhcp_param *paramp;

    paramp = pLeaseData->dhcpcParam;

    pDhcpcReqSpec->ipaddr.s_addr = paramp->yiaddr.s_addr;
    pDhcpcReqSpec->srvaddr.s_addr = paramp->server_id.s_addr;
    if (pLeaseData->leaseReqSpec.clid != NULL)
        pDhcpcReqSpec->clid = pLeaseData->leaseReqSpec.clid;
    else
        pDhcpcReqSpec->clid = NULL;
    if (pLeaseData->leaseReqSpec.dhcp_errmsg != NULL)
        pDhcpcReqSpec->dhcp_errmsg = pLeaseData->leaseReqSpec.dhcp_errmsg;
    else
        {
        inet_ntoa_b (paramp->yiaddr, output);
        sprintf (errmsg, "Releasing the current IP address (%s).", output);
        pDhcpcReqSpec->dhcp_errmsg = errmsg;
        }
    return;
    }

/*******************************************************************************
*
* make_discover - construct a DHCP discover message
*
* This routine constructs an outgoing UDP/IP message containing the values
* required to broadcast a lease request. The <xidFlag> indicates whether
* a transaction ID should be generated. Because multiple leases are supported,
* the contents of the transmit buffer are not guaranteed to remain unchanged.
* Therefore, each message must be rebuilt before it is sent. However, the
* transaction ID must not be changed after the initial transmission.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void make_discover
    (
    LEASE_DATA * 	pLeaseData, 	/* lease-specific data structures */
    BOOL 		xidFlag 	/* generate a new transaction ID? */
    )
    {
    int i = 0;
    int offopt = 0;                   /* offset in options field */
    u_long tmpul = 0;
    u_short tmpus = 0;
    struct dhcp_reqspec * 	pReqSpec;

    pReqSpec = &pLeaseData->leaseReqSpec;

    /* construct dhcp part */

    bzero (sbuf.buf, sbuf.size + DHCPC_OFF);
    dhcpcMsgOut.dhcp->op = BOOTREQUEST;
    dhcpcMsgOut.dhcp->htype = pLeaseData->ifData.haddr.htype;
    dhcpcMsgOut.dhcp->hlen = pLeaseData->ifData.haddr.hlen;

    if (xidFlag)
        pLeaseData->xid = generate_xid (&pLeaseData->ifData);
    dhcpcMsgOut.dhcp->xid = pLeaseData->xid;

    bcopy (pLeaseData->ifData.haddr.haddr, dhcpcMsgOut.dhcp->chaddr,
           dhcpcMsgOut.dhcp->hlen);

    /* insert magic cookie */

    bcopy ( (char *)dhcpCookie, dhcpcMsgOut.dhcp->options, MAGIC_LEN);
    offopt = MAGIC_LEN;

    /* insert message type */

    dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_MSGTYPE_TAG;
    dhcpcMsgOut.dhcp->options [offopt++] = 1;
    dhcpcMsgOut.dhcp->options [offopt++] = DHCPDISCOVER;

    /* insert class identifier */

    if (pReqSpec->classId != NULL && CHKOFF (pReqSpec->classId->len))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_CLASS_ID_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->classId->len;
        bcopy (pReqSpec->classId->id, &dhcpcMsgOut.dhcp->options [offopt],
               pReqSpec->classId->len);
        offopt += pReqSpec->classId->len;
        }

    /* insert client identifier */

    if (pReqSpec->clid != NULL && CHKOFF (pReqSpec->clid->len + 1))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_CLIENT_ID_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->clid->len + 1;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->clid->type;
        bcopy (pReqSpec->clid->id, &dhcpcMsgOut.dhcp->options [offopt],
               pReqSpec->clid->len);
        offopt += pReqSpec->clid->len;
        }

    /* insert host name */

    if (pReqSpec->hostName != NULL &&
            CHKOFF (strlen(pReqSpec->hostName)))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_HOSTNAME_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = strlen(pReqSpec->hostName);
        bcopy (pReqSpec->hostName, &dhcpcMsgOut.dhcp->options [offopt],
               strlen(pReqSpec->hostName));
        offopt += strlen(pReqSpec->hostName);
        }

    /* insert requesting lease */

    if (pReqSpec->lease != 0 && CHKOFF (4))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_LEASE_TIME_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = 4;
        tmpul = htonl (pReqSpec->lease);
        bcopy ( (char *)&tmpul, &dhcpcMsgOut.dhcp->options [offopt], 4);
        offopt += 4;
        }

    /* insert requesting ipaddr */

    if (pReqSpec->ipaddr.s_addr != 0 && CHKOFF (4))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_REQUEST_IPADDR_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = 4;
        bcopy ( (char *)&pReqSpec->ipaddr.s_addr,
                &dhcpcMsgOut.dhcp->options [offopt], 4);
        offopt += 4;
        }

    /* insert Maximum DHCP message size */

    if (CHKOFF (2))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_MAXMSGSIZE_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = 2;
        tmpus = htons (DFLTDHCPLEN + UDPHL + IPHL);
        bcopy ( (char *)&tmpus, &dhcpcMsgOut.dhcp->options [offopt], 2);
        offopt += 2;
        }

    /* if necessary, insert request list */

    if (pReqSpec->reqlist.len != 0 && CHKOFF (pReqSpec->reqlist.len))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_REQ_LIST_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->reqlist.len;
        bcopy (pReqSpec->reqlist.list, &dhcpcMsgOut.dhcp->options [offopt],
	       pReqSpec->reqlist.len);
        offopt += pReqSpec->reqlist.len;
        }

    /* insert any vendor-specific information */

    if (pReqSpec->vendlist.len != 0 && CHKOFF (pReqSpec->vendlist.len))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_VENDOR_SPEC_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->vendlist.len;
        bcopy (pReqSpec->vendlist.list, &dhcpcMsgOut.dhcp->options [offopt],
               pReqSpec->vendlist.len);
        offopt += pReqSpec->vendlist.len;
        }

    dhcpcMsgOut.dhcp->options [offopt] = _DHCP_END_TAG;

    /* make udp part */

    /* fill udp header */

    dhcpcMsgOut.udp->uh_sport = dhcpc_port;
    dhcpcMsgOut.udp->uh_dport = dhcps_port;
    dhcpcMsgOut.udp->uh_ulen = htons (DFLTDHCPLEN + UDPHL);

    /* fill pseudo udp header */

    spudph.srcip.s_addr = 0;
    spudph.dstip.s_addr = 0xffffffff;
    spudph.zero = 0;
    spudph.prto = IPPROTO_UDP;
    spudph.ulen = dhcpcMsgOut.udp->uh_ulen;

    /* make ip part */

    /* fill ip header */

#if BSD<44
    dhcpcMsgOut.ip->ip_v_hl = 0;
    dhcpcMsgOut.ip->ip_v_hl = IPVERSION << 4;
    dhcpcMsgOut.ip->ip_v_hl |= IPHL >> 2;
#else
    dhcpcMsgOut.ip->ip_v = IPVERSION;
    dhcpcMsgOut.ip->ip_hl = IPHL >> 2;
#endif
    dhcpcMsgOut.ip->ip_tos = 0;
    dhcpcMsgOut.ip->ip_len = htons (DFLTDHCPLEN + UDPHL + IPHL);
    tmpul = generate_xid (&pLeaseData->ifData);
    tmpul += (tmpul >> 16);
    dhcpcMsgOut.ip->ip_id = /* htons ( (u_short) (~tmpul));*/
    dhcpcMsgOut.ip->ip_id = (u_short) (~tmpul);
    dhcpcMsgOut.ip->ip_off = htons (IP_DF);                        /* XXX */
    dhcpcMsgOut.ip->ip_ttl = 0x20;                                 /* XXX */
    dhcpcMsgOut.ip->ip_p = IPPROTO_UDP;
    dhcpcMsgOut.ip->ip_src.s_addr = 0;
    dhcpcMsgOut.ip->ip_dst.s_addr = 0xffffffff;
    dhcpcMsgOut.ip->ip_sum = 0;

#if BSD<44
    /* tmpus = checksum ( (u_short *)snd.ip, (snd.ip->ip_v_hl & 0xf) << 2);
    snd.ip->ip_sum = htons (tmpus); */
  dhcpcMsgOut.ip->ip_sum = checksum ( (u_short *)dhcpcMsgOut.ip,
                                      (dhcpcMsgOut.ip->ip_v_hl & 0xf) << 2);
#else
    /* tmpus = checksum ( (u_short *)snd.ip, snd.ip->ip_hl << 2);
    snd.ip->ip_sum = htons (tmpus); */
    dhcpcMsgOut.ip->ip_sum = checksum ( (u_short *)dhcpcMsgOut.ip,
                                        dhcpcMsgOut.ip->ip_hl << 2);
#endif

    /* make ether part */

    /* fill ether frame header */
    for (i = 0; i < 6; i++)
        {
        dhcpcMsgOut.ether->ether_dhost[i] = 0xff;
        dhcpcMsgOut.ether->ether_shost[i] = pLeaseData->ifData.haddr.haddr[i];
        }
    dhcpcMsgOut.ether->ether_type = ETHERTYPE_IP;

    return;
    }

/*******************************************************************************
*
* make_request - construct a DHCP request message
*
* This routine constructs an outgoing UDP/IP message containing the values
* required to request a lease. There is some variation in contents depending
* on whether the request is meant to establish a lease, renew a lease, or
* verify an existing lease. Also, when constructing a lease renewal message,
* the routine exits before any headers are are added so that the message may
* be sent with send_unicast().
* .IP
* The <xidFlag> indicates whether a transaction ID should be generated.
* Because multiple leases are supported, the contents of the transmit buffer
* are not guaranteed to remain unchanged. Therefore, each message must be
* rebuilt before it is sent. However, the transaction ID must not be changed
* after the initial transmission.
*
* RETURNS: 0 if constructed succesfully, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int make_request
    (
    LEASE_DATA * 	pLeaseData, 	/* lease-specific data structures */
    int type,
    BOOL 		xidFlag 	/* generate a new transaction ID? */
    )
    {
    int i = 0;
    int offopt = 0;                   /* offset in options field */
    u_long tmpul = 0;
    u_short tmpus = 0;
    struct dhcp_reqspec * 	pReqSpec = NULL;
    struct dhcp_param * 	paramp = NULL;

    pReqSpec = &pLeaseData->leaseReqSpec;
    paramp = pLeaseData->dhcpcParam;

    /* construct dhcp part */

    bzero (sbuf.buf, sbuf.size + DHCPC_OFF);
    dhcpcMsgOut.dhcp->op = BOOTREQUEST;
    dhcpcMsgOut.dhcp->htype = pLeaseData->ifData.haddr.htype;
    dhcpcMsgOut.dhcp->hlen = pLeaseData->ifData.haddr.hlen;

    if (xidFlag)
        pLeaseData->xid = generate_xid (&pLeaseData->ifData);
    dhcpcMsgOut.dhcp->xid = pLeaseData->xid;

    dhcpcMsgOut.dhcp->secs = 0;
    if (type == REQUESTING || type == REBOOTING)
        dhcpcMsgOut.dhcp->ciaddr.s_addr = 0;
    else
        dhcpcMsgOut.dhcp->ciaddr.s_addr = paramp->yiaddr.s_addr;

    bcopy (pLeaseData->ifData.haddr.haddr, dhcpcMsgOut.dhcp->chaddr,
           dhcpcMsgOut.dhcp->hlen);

    /* insert magic cookie */

    bcopy ( (char *)dhcpCookie, dhcpcMsgOut.dhcp->options, MAGIC_LEN);
    offopt = MAGIC_LEN;

    /* insert message type */
    dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_MSGTYPE_TAG;
    dhcpcMsgOut.dhcp->options [offopt++] = 1;
    dhcpcMsgOut.dhcp->options [offopt++] = DHCPREQUEST;

    /* insert class identifier */

    if (pReqSpec->classId != NULL && CHKOFF (pReqSpec->classId->len))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_CLASS_ID_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->classId->len;
        bcopy (pReqSpec->classId->id, &dhcpcMsgOut.dhcp->options [offopt],
               pReqSpec->classId->len);
        offopt += pReqSpec->classId->len;
        }

    /* insert client identifier */

    if (pReqSpec->clid != NULL && CHKOFF (pReqSpec->clid->len + 1))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_CLIENT_ID_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->clid->len + 1;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->clid->type;
        bcopy (pReqSpec->clid->id, &dhcpcMsgOut.dhcp->options [offopt],
               pReqSpec->clid->len);
        offopt += pReqSpec->clid->len;
        }

    /* insert host name */

    if (pReqSpec->hostName != NULL && CHKOFF (strlen(pReqSpec->hostName)))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_HOSTNAME_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = strlen(pReqSpec->hostName);
        bcopy (pReqSpec->hostName, &dhcpcMsgOut.dhcp->options [offopt],
               strlen(pReqSpec->hostName));
        offopt += strlen(pReqSpec->hostName);
        }

    /* insert requesting lease */

    if (type != VERIFYING && pReqSpec->lease != 0 && CHKOFF (4))
        {
        dhcpcMsgOut.dhcp->options[offopt++] = _DHCP_LEASE_TIME_TAG;
        dhcpcMsgOut.dhcp->options[offopt++] = 4;
        tmpul = htonl (pReqSpec->lease);
        bcopy ( (char *)&tmpul, &dhcpcMsgOut.dhcp->options[offopt], 4);
        offopt += 4;
        }

    /* insert requesting ipaddr */

    if (type == REQUESTING || type == REBOOTING)
        {
        if (paramp->yiaddr.s_addr != 0 && CHKOFF(4))
            {
            dhcpcMsgOut.dhcp->options[offopt++] = _DHCP_REQUEST_IPADDR_TAG;
            dhcpcMsgOut.dhcp->options[offopt++] = 4;
            bcopy ( (char *)&paramp->yiaddr.s_addr,
                   &dhcpcMsgOut.dhcp->options [offopt], 4);
            offopt += 4;
            }
        else
            return (-1);
        }

    /* insert server identifier */

    if (type == REQUESTING)
        {
        if (paramp->server_id.s_addr == 0)
            {
             return(-1);
              }
        if (CHKOFF (4))
            {
            dhcpcMsgOut.dhcp->options[offopt++] = _DHCP_SERVER_ID_TAG;
            dhcpcMsgOut.dhcp->options[offopt++] = 4;
            bcopy ( (char *)&paramp->server_id.s_addr,
                   &dhcpcMsgOut.dhcp->options [offopt], 4);
            offopt += 4;
            }
        }

    /* insert Maximum DHCP message size */

    if (CHKOFF (2))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_MAXMSGSIZE_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = 2;
        tmpus = htons (DFLTDHCPLEN + UDPHL + IPHL);
        bcopy ( (char *)&tmpus, &dhcpcMsgOut.dhcp->options [offopt], 2);
        offopt += 2;
        }

    /* if necessary, insert request list */

    if (pReqSpec->reqlist.len != 0 && CHKOFF (pReqSpec->reqlist.len))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_REQ_LIST_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->reqlist.len;
        bcopy (pReqSpec->reqlist.list, &dhcpcMsgOut.dhcp->options [offopt],
	       pReqSpec->reqlist.len);
        offopt += pReqSpec->reqlist.len;
        }

    /* insert any vendor-specific information */
    if (pReqSpec->vendlist.len != 0 && CHKOFF (pReqSpec->vendlist.len))
        {
        dhcpcMsgOut.dhcp->options [offopt++] = _DHCP_VENDOR_SPEC_TAG;
        dhcpcMsgOut.dhcp->options [offopt++] = pReqSpec->vendlist.len;
        bcopy (pReqSpec->vendlist.list, &dhcpcMsgOut.dhcp->options[offopt],
               pReqSpec->vendlist.len);
        offopt += pReqSpec->vendlist.len;
        }

    dhcpcMsgOut.dhcp->options[offopt] = _DHCP_END_TAG;

    if (type == RENEWING)    /* RENEWING is unicast with the normal socket */
        return(0);

    /* make udp part */

    /* fill udp header */

    dhcpcMsgOut.udp->uh_sport = dhcpc_port;
    dhcpcMsgOut.udp->uh_dport = dhcps_port;
    dhcpcMsgOut.udp->uh_ulen = htons (DFLTDHCPLEN + UDPHL);

    /* fill pseudo udp header */

    spudph.zero = 0;
    spudph.prto = IPPROTO_UDP;
    spudph.ulen = dhcpcMsgOut.udp->uh_ulen;

    /* make ip part */

    /* fill ip header */

#if BSD<44
    dhcpcMsgOut.ip->ip_v_hl = 0;
    dhcpcMsgOut.ip->ip_v_hl = IPVERSION << 4;
    dhcpcMsgOut.ip->ip_v_hl |= IPHL >> 2;
#else
    dhcpcMsgOut.ip->ip_v = IPVERSION;
    dhcpcMsgOut.ip->ip_hl = IPHL >> 2;
#endif
    dhcpcMsgOut.ip->ip_tos = 0;
    dhcpcMsgOut.ip->ip_len = htons (DFLTDHCPLEN + UDPHL + IPHL);
    tmpul = generate_xid (&pLeaseData->ifData);
    tmpul += (tmpul >> 16);
    dhcpcMsgOut.ip->ip_id = (u_short) (~tmpul);
    dhcpcMsgOut.ip->ip_off = htons (IP_DF);                        /* XXX */
    dhcpcMsgOut.ip->ip_ttl = 0x20;                                 /* XXX */
    dhcpcMsgOut.ip->ip_p = IPPROTO_UDP;

    switch (type)
        {
        case REQUESTING:     /* fall-through */
        case REBOOTING:      /* fall-through */
        case VERIFYING:
            dhcpcMsgOut.ip->ip_src.s_addr = spudph.srcip.s_addr = 0;
            dhcpcMsgOut.ip->ip_dst.s_addr = spudph.dstip.s_addr = 0xffffffff;
            break;
        case REBINDING:
            dhcpcMsgOut.ip->ip_src.s_addr = spudph.srcip.s_addr =
                                            paramp->yiaddr.s_addr;
            dhcpcMsgOut.ip->ip_dst.s_addr = spudph.dstip.s_addr = 0xffffffff;
            break;
        }
    dhcpcMsgOut.ip->ip_sum = 0;
#if BSD<44
    dhcpcMsgOut.ip->ip_sum = checksum ( (u_short *)dhcpcMsgOut.ip,
                                        (dhcpcMsgOut.ip->ip_v_hl & 0xf) << 2);
#else
    dhcpcMsgOut.ip->ip_sum = checksum ( (u_short *)dhcpcMsgOut.ip,
                                        dhcpcMsgOut.ip->ip_hl << 2);
#endif

    /* make ether part */

    /* fill ether frame header */

    for (i = 0; i < 6; i++)
        {
        dhcpcMsgOut.ether->ether_dhost[i] = 0xff;
        dhcpcMsgOut.ether->ether_shost[i] = pLeaseData->ifData.haddr.haddr[i];
        }
    dhcpcMsgOut.ether->ether_type = ETHERTYPE_IP;

    return (0);
    }

/*******************************************************************************
*
* send_unicast - send a DHCP message via unicast
*
* This routine sends a previously constructed DHCP message to the DHCP
* server port on the specified IP address. It is used to send a request
* for lease renewal or a DHCP release message to the appropriate DHCP server.
*
* RETURNS: 0 if message sent, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int send_unicast
    (
    struct in_addr *dstip,
    struct dhcp *sdhcp
    )
    {
    int sockfd = -1;
    struct sockaddr_in dst;
    struct msghdr msg;
    struct iovec bufvec[1];
    int bufsize = DFLTDHCPLEN;
    int status;

    if (sockfd == -1)
        {
        struct sockaddr_in myaddr;

        if ( (sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
            {
            printf ("socket error %d (%x).\n", sockfd, errno);
            return (-1);
            }

        bzero ( (char *)&myaddr, sizeof (myaddr));
        myaddr.sin_family = AF_INET;
        myaddr.sin_addr.s_addr = htonl (INADDR_ANY);
        myaddr.sin_port = dhcpc_port;

        status = bind (sockfd, (struct sockaddr *) &myaddr, sizeof (myaddr));
        if (status < 0)
            {
            printf ("bind error %d (%x).\n", status, errno);
            close (sockfd);
            sockfd = -1;
            return (-1);
            }
        status = setsockopt (sockfd, SOL_SOCKET, SO_SNDBUF, (char *)&bufsize,
		        sizeof (bufsize));
        if (status < 0)
            {
            printf ("setsockopt error %d (%x).\n", status, errno);
            close (sockfd);
            sockfd = -1;
            return (-1);
            }

        }

    bzero ( (char *)&dst, sizeof (dst));
    dst.sin_family = AF_INET;
    dst.sin_addr.s_addr = dstip->s_addr;
    dst.sin_port = dhcps_port;

    bufvec[0].iov_base = (char *)sdhcp;
    bufvec[0].iov_len = bufsize;

    bzero ( (char *)&msg, sizeof (msg));
    msg.msg_name = (caddr_t)&dst;
    msg.msg_namelen = sizeof (dst);
    msg.msg_iov = bufvec;
    msg.msg_iovlen = 1;
    status = sendmsg (sockfd, &msg, 0);
    if (status < 0)
        {
        printf ("sendmsg error %d (%x).\n", status, errno);
        close (sockfd);
        sockfd = -1;
        return (-1);
        }
    close (sockfd);
    sockfd = -1;
    return(0);
    }

/*******************************************************************************
*
* handle_ip - process DHCP options containing a single IP address
*
* This routine extracts the IP address from the given option body and
* copies it to the appropriate field in the parameters structure. It is
* called by the dhcp_msgtoparam() conversion routine when specified by
* the handle_param[] global array.
*
* RETURNS: 0 if extraction successful, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int handle_ip
    (
    char *buf,
    struct dhcp_param *param
    )
    {
    struct in_addr *addr = NULL;
    char option;

    option = *buf;

    /* Set the appropriate pointers to access allocated memory. */

    if (option == _DHCP_SERVER_ID_TAG)
        {
        addr = &param->server_id;
        }
    else if (option == _DHCP_ROUTER_SOLICIT_TAG)
        addr = &param->router_solicit;
    else
        {
        addr = (struct in_addr *)calloc (1, sizeof (struct in_addr));
        if (addr == NULL)
            return (-1);

        switch (option)
            {
            case _DHCP_SUBNET_MASK_TAG:
                if (param->subnet_mask != NULL)
                    free (param->subnet_mask);
                param->subnet_mask = addr;
                break;
            case _DHCP_SWAP_SERVER_TAG:
                if (param->swap_server != NULL)
                    free (param->swap_server);
                param->swap_server = addr;
                break;
            case _DHCP_BRDCAST_ADDR_TAG:
                if (param->brdcast_addr != NULL)
                    free (param->brdcast_addr);
                param->brdcast_addr = addr;
                break;
            default:
                free (addr);
                return (EINVAL);
            }
        }

    bcopy (OPTBODY (buf), (char *)addr, DHCPOPTLEN (buf));
    return (0);
    }


/*******************************************************************************
*
* handle_num - process DHCP options containing a single numeric value
*
* This routine extracts the numeric value from the given option body and
* stores it in the appropriate field in the parameters structure. It is
* called by the dhcp_msgtoparam() conversion routine when specified by
* the handle_param[] global array.
*
* RETURNS: 0 if extraction successful, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int handle_num
    (
    char *buf,
    struct dhcp_param *param
    )
    {
    char   charnum = 0;
    short  shortnum = 0;
    long   longnum = 0;

    switch (DHCPOPTLEN (buf))
        {
        case 1:
            charnum = *OPTBODY (buf);
            break;
        case 2:
            shortnum = GETHS (OPTBODY (buf));
            break;
        case 4:
            longnum = GETHL (OPTBODY (buf));
            break;
        default:
            return (-1);
        }

    switch (*buf)
        {
        case _DHCP_TIME_OFFSET_TAG:
            param->time_offset = longnum;
            break;
        case _DHCP_BOOTSIZE_TAG:
            param->bootsize = (unsigned short)shortnum;
            break;
        case _DHCP_MAX_DGRAM_SIZE_TAG:
            param->max_dgram_size = (unsigned short)shortnum;
            break;
        case _DHCP_DEFAULT_IP_TTL_TAG:
            param->default_ip_ttl = (unsigned char)charnum;
            break;
        case _DHCP_MTU_AGING_TIMEOUT_TAG:
            param->mtu_aging_timeout = (unsigned long)longnum;
            break;
        case _DHCP_IF_MTU_TAG:
            param->intf_mtu = (unsigned short)shortnum;
            break;
        case _DHCP_ARP_CACHE_TIMEOUT_TAG:
            param->arp_cache_timeout = (unsigned long)longnum;
            break;
        case _DHCP_DEFAULT_TCP_TTL_TAG:
            param->default_tcp_ttl = (unsigned char)charnum;
            break;
        case _DHCP_KEEPALIVE_INTERVAL_TAG:
            param->keepalive_inter = (unsigned long)longnum;
            break;
        case _DHCP_NB_NODETYPE_TAG:
            param->nb_nodetype = (unsigned)charnum;
            break;
        case _DHCP_LEASE_TIME_TAG:
            param->lease_duration = (unsigned long)longnum;
            break;
        case _DHCP_T1_TAG:
            param->dhcp_t1 = (unsigned long)longnum;
            break;
        case _DHCP_T2_TAG:
            param->dhcp_t2 = (unsigned long)longnum;
            break;
        default:
            return (EINVAL);
        }

    return(0);
    }

/*******************************************************************************
*
* handle_ips - process DHCP options containing multiple IP addresses
*
* This routine extracts the IP addresses from the given option body and
* copies them to the appropriate field in the parameters structure. It is
* called by the dhcp_msgtoparam() conversion routine when specified by
* the handle_param[] global array.
*
* RETURNS: 0 if extraction successful, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int handle_ips
    (
    char *buf,
    struct dhcp_param *param
    )
    {
    struct in_addr  *addr = NULL;
    struct in_addrs *addrs = NULL;
    unsigned char  num = 0;

    num = DHCPOPTLEN (buf) / 4;

    addr = (struct in_addr *)calloc ( (int)num, sizeof (struct in_addr));
    if (addr  == NULL)
        return(-1);

    addrs = (struct in_addrs *)calloc (1, sizeof(struct in_addrs));
    if (addrs  == NULL)
        {
        free (addr);
        return (-1);
        }

    switch (*buf)
        {
        case _DHCP_ROUTER_TAG:
            if (param->router != NULL)
                {
                if (param->router->addr != NULL)
                    free (param->router->addr);
                free (param->router);
                }
            param->router = addrs;
            param->router->num = num;
            param->router->addr = addr;
            break;
        case _DHCP_TIME_SERVER_TAG:
            if (param->time_server != NULL)
                {
                if (param->time_server->addr != NULL)
                    free (param->time_server->addr);
                free (param->time_server);
                }
            param->time_server = addrs;
            param->time_server->num = num;
            param->time_server->addr = addr;
            break;
        case _DHCP_NAME_SERVER_TAG:
            if (param->name_server != NULL)
                {
                if (param->name_server->addr != NULL)
                    free (param->name_server->addr);
                free (param->name_server);
                }
            param->name_server = addrs;
            param->name_server->num = num;
            param->name_server->addr = addr;
            break;
        case _DHCP_DNS_SERVER_TAG:
            if (param->dns_server != NULL)
                {
                if (param->dns_server->addr != NULL)
                    free (param->dns_server->addr);
                free (param->dns_server);
                }
            param->dns_server = addrs;
            param->dns_server->num = num;
            param->dns_server->addr = addr;
            break;
        case _DHCP_LOG_SERVER_TAG:
            if (param->log_server != NULL)
                {
                if (param->log_server->addr != NULL)
                    free (param->log_server->addr);
                free (param->log_server);
                }
            param->log_server = addrs;
            param->log_server->num = num;
            param->log_server->addr = addr;
            break;
        case _DHCP_COOKIE_SERVER_TAG:
            if (param->cookie_server != NULL)
                {
                if (param->cookie_server->addr != NULL)
                    free (param->cookie_server->addr);
                free (param->cookie_server);
                }
            param->cookie_server = addrs;
            param->cookie_server->num = num;
            param->cookie_server->addr = addr;
            break;
        case _DHCP_LPR_SERVER_TAG:
            if (param->lpr_server != NULL)
                {
                if (param->lpr_server->addr != NULL)
                    free (param->lpr_server->addr);
                free (param->lpr_server);
                }
            param->lpr_server = addrs;
            param->lpr_server->num = num;
            param->lpr_server->addr = addr;
            break;
        case _DHCP_IMPRESS_SERVER_TAG:
            if (param->impress_server != NULL)
                {
                if (param->impress_server->addr != NULL)
                    free (param->impress_server->addr);
                free (param->impress_server);
                }
            param->impress_server = addrs;
            param->impress_server->num = num;
            param->impress_server->addr = addr;
            break;
        case _DHCP_RLS_SERVER_TAG:
            if (param->rls_server != NULL)
                {
                if (param->rls_server->addr != NULL)
                    free (param->rls_server->addr);
                free (param->rls_server);
                }
            param->rls_server = addrs;
            param->rls_server->num = num;
            param->rls_server->addr = addr;
            break;
        case _DHCP_NIS_SERVER_TAG:
            if (param->nis_server != NULL)
                {
                if (param->nis_server->addr != NULL)
                    free (param->nis_server->addr);
                free (param->nis_server);
                }
            param->nis_server = addrs;
            param->nis_server->num = num;
            param->nis_server->addr = addr;
            break;
        case _DHCP_NTP_SERVER_TAG:
            if (param->ntp_server != NULL)
                {
                if (param->ntp_server->addr != NULL)
                    free (param->ntp_server->addr);
                free (param->ntp_server);
                }
            param->ntp_server = addrs;
            param->ntp_server->num = num;
            param->ntp_server->addr = addr;
            break;
        case _DHCP_NBN_SERVER_TAG:
            if (param->nbn_server != NULL)
                {
                if (param->nbn_server->addr != NULL)
                    free (param->nbn_server->addr);
                free (param->nbn_server);
                }
            param->nbn_server = addrs;
            param->nbn_server->num = num;
            param->nbn_server->addr = addr;
            break;
        case _DHCP_NBDD_SERVER_TAG:
            if (param->nbdd_server != NULL)
                {
                if (param->nbdd_server->addr != NULL)
                    free (param->nbdd_server->addr);
                free (param->nbdd_server);
                }
            param->nbdd_server = addrs;
            param->nbdd_server->num = num;
            param->nbdd_server->addr = addr;
            break;
        case _DHCP_XFONT_SERVER_TAG:
            if (param->xfont_server != NULL)
                {
                if (param->xfont_server->addr != NULL)
                    free (param->xfont_server->addr);
                free (param->xfont_server);
                }
            param->xfont_server = addrs;
            param->xfont_server->num = num;
            param->xfont_server->addr = addr;
            break;
        case _DHCP_XDISPLAY_MANAGER_TAG:
            if (param->xdisplay_manager != NULL)
                {
                if (param->xdisplay_manager->addr != NULL)
                    free (param->xdisplay_manager->addr);
                free (param->xdisplay_manager);
                }
            param->xdisplay_manager = addrs;
            param->xdisplay_manager->num = num;
            param->xdisplay_manager->addr = addr;
            break;
        case _DHCP_NISP_SERVER_TAG:
            if (param->nisp_server != NULL)
                {
                if (param->nisp_server->addr != NULL)
                    free (param->nisp_server->addr);
                free (param->nisp_server);
                }
            param->nisp_server = addrs;
            param->nisp_server->num = num;
            param->nisp_server->addr = addr;
            break;
        case _DHCP_MOBILEIP_HA_TAG:
            if (param->mobileip_ha != NULL)
                {
                if (param->mobileip_ha->addr != NULL)
                    free (param->mobileip_ha->addr);
                free (param->mobileip_ha);
                }
            param->mobileip_ha = addrs;
            param->mobileip_ha->num = num;
            param->mobileip_ha->addr = addr;
            break;
        case _DHCP_SMTP_SERVER_TAG:
            if (param->smtp_server != NULL)
                {
                if (param->smtp_server->addr != NULL)
                    free (param->smtp_server->addr);
                free (param->smtp_server);
                }
            param->smtp_server = addrs;
            param->smtp_server->num = num;
            param->smtp_server->addr = addr;
            break;
        case _DHCP_POP3_SERVER_TAG:
            if (param->pop3_server != NULL)
                {
                if (param->pop3_server->addr != NULL)
                    free (param->pop3_server->addr);
                free (param->pop3_server);
                }
            param->pop3_server = addrs;
            param->pop3_server->num = num;
            param->pop3_server->addr = addr;
            break;
        case _DHCP_NNTP_SERVER_TAG:
            if (param->nntp_server != NULL)
                {
                if (param->nntp_server->addr != NULL)
                    free (param->nntp_server->addr);
                free (param->nntp_server);
                }
            param->nntp_server = addrs;
            param->nntp_server->num = num;
            param->nntp_server->addr = addr;
            break;
        case _DHCP_DFLT_WWW_SERVER_TAG:
            if (param->dflt_www_server != NULL)
                {
                if (param->dflt_www_server->addr != NULL)
                    free (param->dflt_www_server->addr);
                free (param->dflt_www_server);
                }
            param->dflt_www_server = addrs;
            param->dflt_www_server->num = num;
            param->dflt_www_server->addr = addr;
            break;
        case _DHCP_DFLT_FINGER_SERVER_TAG:
            if (param->dflt_finger_server != NULL)
                {
                if (param->dflt_finger_server->addr != NULL)
	            free (param->dflt_finger_server->addr);
                free (param->dflt_finger_server);
                }
            param->dflt_finger_server = addrs;
            param->dflt_finger_server->num = num;
            param->dflt_finger_server->addr = addr;
            break;
        case _DHCP_DFLT_IRC_SERVER_TAG:
            if (param->dflt_irc_server != NULL)
                {
                if (param->dflt_irc_server->addr != NULL)
                    free (param->dflt_irc_server->addr);
                free (param->dflt_irc_server);
                }
            param->dflt_irc_server = addrs;
            param->dflt_irc_server->num = num;
            param->dflt_irc_server->addr = addr;
            break;
        case _DHCP_STREETTALK_SERVER_TAG:
            if (param->streettalk_server != NULL)
                {
                if (param->streettalk_server->addr != NULL)
                    free (param->streettalk_server->addr);
                free (param->streettalk_server);
                }
            param->streettalk_server = addrs;
            param->streettalk_server->num = num;
            param->streettalk_server->addr = addr;
            break;
        case _DHCP_STDA_SERVER_TAG:
            if (param->stda_server != NULL)
                {
                if (param->stda_server->addr != NULL)
                    free (param->stda_server->addr);
                free (param->stda_server);
                }
            param->stda_server = addrs;
            param->stda_server->num = num;
            param->stda_server->addr = addr;
            break;
        default:
            free (addr);
            free (addrs);
            return (EINVAL);
        }
    bcopy (OPTBODY (buf), (char *)addr, DHCPOPTLEN (buf));
    return (0);
    }

/*******************************************************************************
*
* handle_str - process DHCP options containing NVT ASCII strings
*
* This routine extracts the NVT ASCII string from the given option body and
* copies it to the appropriate field in the parameters structure. It is
* called by the dhcp_msgtoparam() conversion routine when specified by
* the handle_param[] global array.
*
* RETURNS: 0 if extraction successful, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int handle_str
    (
    char *buf,
    struct dhcp_param *param
    )
    {
    char *str = NULL;

    str = calloc (1, (DHCPOPTLEN (buf) + 1));  /* +1 for null terminator. */
    if (str == NULL)
        return (-1);

    switch (*buf)
        {
        case _DHCP_HOSTNAME_TAG:
            if (param->hostname != NULL)
                free (param->hostname);
            param->hostname = str;
            break;
        case _DHCP_MERIT_DUMP_TAG:
            if (param->merit_dump != NULL)
                free (param->merit_dump);
            param->merit_dump = str;
            break;
        case _DHCP_DNS_DOMAIN_TAG:
            if (param->dns_domain != NULL)
                free (param->dns_domain);
            param->dns_domain = str;
            break;
        case _DHCP_ROOT_PATH_TAG:
            if (param->root_path != NULL)
                free (param->root_path);
            param->root_path = str;
            break;
        case _DHCP_EXTENSIONS_PATH_TAG:
            if (param->extensions_path != NULL)
                free (param->extensions_path);
            param->extensions_path = str;
            break;
        case _DHCP_NIS_DOMAIN_TAG:
            if (param->nis_domain != NULL)
                free (param->nis_domain);
            param->nis_domain = str;
            break;
        case _DHCP_NB_SCOPE_TAG:
            if (param->nb_scope != NULL)
                free (param->nb_scope);
            param->nb_scope = str;
            break;
        case _DHCP_ERRMSG_TAG:
            if (param->errmsg != NULL)
                free (param->errmsg);
            param->errmsg = str;
            break;
        case _DHCP_NISP_DOMAIN_TAG:
            if (param->nisp_domain != NULL)
                free (param->nisp_domain);
            param->nisp_domain = str;
            break;
        case _DHCP_TFTP_SERVERNAME_TAG:
            if (param->temp_sname != NULL)
                free (param->temp_sname);
            param->temp_sname = str;
            break;
        case _DHCP_BOOTFILE_TAG:
            if (param->temp_file != NULL)
                free (param->temp_file);
            param->temp_file = str;
            break;
        default:
            free (str);
            return (EINVAL);
        }
    bcopy (OPTBODY (buf), str, DHCPOPTLEN (buf));
    str [(int)(DHCPOPTLEN (buf))] = '\0';
    return (0);
    }

/*******************************************************************************
*
* handle_bool - process DHCP options containing boolean values
*
* This routine extracts the boolean value from the given option body and
* stores it in the appropriate field in the parameters structure. It is
* called by the dhcp_msgtoparam() conversion routine when specified by
* the handle_param[] global array.
*
* RETURNS: 0 if extraction successful, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int handle_bool
    (
    char *buf,
    struct dhcp_param *param
    )
    {
    switch (*buf)
        {
        case _DHCP_IP_FORWARD_TAG:
            param->ip_forward = *OPTBODY (buf);
            break;
        case _DHCP_NONLOCAL_SRCROUTE_TAG:
            param->nonlocal_srcroute = *OPTBODY (buf);
            break;
        case _DHCP_ALL_SUBNET_LOCAL_TAG:
            param->all_subnet_local = *OPTBODY (buf);
            break;
        case _DHCP_MASK_DISCOVER_TAG:
            param->mask_discover = *OPTBODY (buf);
            break;
        case _DHCP_MASK_SUPPLIER_TAG:
            param->mask_supplier = *OPTBODY (buf);
            break;
        case _DHCP_ROUTER_DISCOVER_TAG:
            param->router_discover = *OPTBODY (buf);
            break;
        case _DHCP_TRAILER_TAG:
            param->trailer = *OPTBODY (buf);
            break;
        case _DHCP_ETHER_ENCAP_TAG:
            param->ether_encap = *OPTBODY (buf);
            break;
        case _DHCP_KEEPALIVE_GARBAGE_TAG:
            param->keepalive_garba = *OPTBODY (buf);
            break;
        default:
            return (EINVAL);
        }
    return (0);
    }

/*******************************************************************************
*
* handle_ippairs - process DHCP options containing multiple IP value pairs
*
* This routine extracts the IP pairs from the given option body and
* stores them in the appropriate field in the parameters structure. It is
* called by the dhcp_msgtoparam() conversion routine when specified by
* the handle_param[] global array.
*
* RETURNS: 0 if extraction successful, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int handle_ippairs
    (
    char *buf,
    struct dhcp_param *param
    )
    {
    struct in_addr  *addr = NULL;
    struct in_addrs *addrs = NULL;
    unsigned char   num = 0;

    num = DHCPOPTLEN(buf) / 4;
    addr = (struct in_addr *)calloc ( (int) num, sizeof (struct in_addr));
    if (addr == NULL)
        return(-1);

    addrs = (struct in_addrs *)calloc (1, sizeof (struct in_addrs));
    if (addrs == NULL)
        {
        free (addr);
        return (-1);
        }

    switch (*buf)
        {
        case _DHCP_POLICY_FILTER_TAG: /* IP address followed by subnet mask. */
            if (param->policy_filter != NULL)
                {
                if (param->policy_filter->addr != NULL)
                    free (param->policy_filter->addr);
                free (param->policy_filter);
                }
            param->policy_filter = addrs;
            param->policy_filter->num = num / 2;
            param->policy_filter->addr = addr;
            break;
        case _DHCP_STATIC_ROUTE_TAG:   /* Destination IP followed by router. */
            if (param->static_route != NULL)
                {
                if (param->static_route->addr != NULL)
                    free (param->static_route->addr);
                free (param->static_route);
                }
            param->static_route = addrs;
            param->static_route->num = num / 2;
            param->static_route->addr = addr;
            break;
        default:
            free (addr);
            free (addrs);
            return (EINVAL);
        }
    bcopy (OPTBODY (buf), (char *)addr, DHCPOPTLEN (buf));
    return (0);
    }

/*******************************************************************************
*
* handle_nums - process DHCP options containing multiple numeric values
*
* This routine extracts the numbers from the given option body and
* stores them in the appropriate field in the parameters structure. It is
* called by the dhcp_msgtoparam() conversion routine when specified by
* the handle_param[] global array.
*
* RETURNS: 0 if extraction successful, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int handle_nums
    (
    char *buf,
    struct dhcp_param *param
    )
    {
    int i = 0;
    int max = 0;

    if (*buf != _DHCP_MTU_PLATEAU_TABLE_TAG)
        return (EINVAL);
    else
        {
        if (param->mtu_plateau_table != NULL)
            {
            if (param->mtu_plateau_table->shortnum != NULL)
	        free (param->mtu_plateau_table->shortnum);
            free (param->mtu_plateau_table);
            }
        param->mtu_plateau_table =
                       (struct u_shorts *)calloc(1, sizeof (struct u_shorts));
        if (param->mtu_plateau_table == NULL)
            return(-1);

        max = param->mtu_plateau_table->num = DHCPOPTLEN (buf) / 2;
        param->mtu_plateau_table->shortnum =
                            (u_short *)calloc (max, sizeof (unsigned short));
        if (param->mtu_plateau_table->shortnum == NULL)
            {
            free (param->mtu_plateau_table);
            return(-1);
            }

        for (i = 0; i < max; i++)
            param->mtu_plateau_table->shortnum[i] = GETHS (&buf [i * 2 + 2]);
        }
    return (0);
    }

/*******************************************************************************
*
* handle_list - process the DHCP option containing vendor specific data
*
* This routine extracts the list of vendor-specific information from the
* given option body and stores it in the appropriate field in the parameters
* structure. It is called by the dhcp_msgtoparam() conversion routine when
* specified by the handle_param[] global array.
*
* RETURNS: 0 if extraction successful, or -1 on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

//SPR#25580
//int handle_list
int handle_vendor
    (
    char *buf,
    struct dhcp_param *param
    )
    {
    if (*buf != _DHCP_VENDOR_SPEC_TAG)
        return (EINVAL);

    if (param->vendlist != NULL)
        free (param->vendlist);

    param->vendlist = calloc (1, sizeof (struct vendor_list));
    if (param->vendlist == NULL)
        return (-1);

    bcopy (OPTBODY (buf), param->vendlist->list, DHCPOPTLEN (buf));
    param->vendlist->len = DHCPOPTLEN (buf);

    return (0);
    }
