#include <linux/fs.h>
#include<linux/slab.h>
#include <net/datapath_proc_api.h>

static void *ltq_seq_start(struct seq_file *s, loff_t *pos)
{
	struct ltq_proc_file_entry *p = s->private;

	if (p->pos < 0)
		return NULL;

	return p;
}

static void *ltq_seq_next(struct seq_file *s, void *v, loff_t *pos)
{
	struct ltq_proc_file_entry *p = s->private;

	*pos = p->pos;

	if (p->pos >= 0)
		return p;
	else
		return NULL;
}

static void ltq_seq_stop(struct seq_file *s, void *v)
{
}

static int ltq_seq_show(struct seq_file *s, void *v)
{
	struct ltq_proc_file_entry *p = s->private;

	if (p->pos >= 0)
		p->pos = p->callback(s, p->pos);

	return 0;
}

static int ltq_proc_open(struct inode *inode, struct file *file);

static const struct seq_operations dp_seq_ops = {
	.start	= ltq_seq_start,
	.next	= ltq_seq_next,
	.stop	= ltq_seq_stop,
	.show	= ltq_seq_show
};

static int ltq_proc_open(struct inode *inode, struct file *file)
{
	struct seq_file *s;
	struct ltq_proc_file_entry *p;
	struct ltq_proc_entry *entry;
	int ret;

	ret = seq_open(file, &dp_seq_ops);
	if (ret)
		return ret;

	s = file->private_data;
	p = (struct ltq_proc_file_entry *)kmalloc(sizeof(*p), GFP_KERNEL);

	if (!p) {
		(void)seq_release(inode, file);
		return -ENOMEM;
	}

	entry = PDE_DATA(inode);

	p->callback = entry->callback;
	if (entry->init_callback)
		p->pos = entry->init_callback();
	else
		p->pos = 0;

	s->private = p;

	return 0;
}

static int ltq_proc_release(struct inode *inode, struct file *file)
{
	struct seq_file *s;

	s = file->private_data;
	kfree(s->private);

	return seq_release(inode, file);
}

static int ltq_seq_single_show(struct seq_file *s, void *v)
{
	struct ltq_proc_entry *p = s->private;

	p->single_callback(s);
	return 0;
}

static int ltq_proc_single_open(struct inode *inode, struct file *file)
{
	return single_open(file, ltq_seq_single_show, PDE_DATA(inode));
}


void ltq_proc_entry_create(struct proc_dir_entry *parent_node,
	struct ltq_proc_entry *proc_entry)
{
	memset(&proc_entry->ops, 0, sizeof(struct file_operations));
	proc_entry->ops.owner = THIS_MODULE;

	if (proc_entry->single_callback) {
		proc_entry->ops.open = ltq_proc_single_open;
		proc_entry->ops.release = single_release;
	} else {
		proc_entry->ops.open = ltq_proc_open;
		proc_entry->ops.release = ltq_proc_release;
	}

	proc_entry->ops.read = seq_read;
	proc_entry->ops.llseek = seq_lseek;
	proc_entry->ops.write = proc_entry->write_callback;
	proc_create_data(proc_entry->name,
	                 (S_IFREG | S_IRUGO),
	                 parent_node, &proc_entry->ops, proc_entry);
}



